import {React, Component, B, refCallback} from '../base';
import propTypes from '../_utils/propTypes';

import {ROBOT_MONTH} from '../../lib/date/formats';
import {getHeight} from '../../lib/dom/dimensions';
import {
    calcSizeCache,
    calcAbsoluteScroll,
    calcRelativeScroll,
} from '../../lib/relativeScrolling';

import Month from '../Month/Month';
import Weekday from '../Weekday/Weekday';
import Weekdays from '../Weekdays/Weekdays';
import CalendarToolbar from '../CalendarToolbar/CalendarToolbar';

const b = B('CalendarContent');

class CalendarContent extends Component {
    componentDidMount() {
        const monthElements = [...this._container.childNodes];

        const containerHeight = getHeight(this._container);
        const monthHeights = monthElements.map(monthElement =>
            getHeight(monthElement, 'padding'),
        );

        this._sizeCache = calcSizeCache(containerHeight, monthHeights);
        this.scrollTo(this.props.scroll);
    }

    componentWillReceiveProps({scroll}) {
        this.scrollTo(scroll);
    }

    shouldComponentUpdate(nextProps) {
        return (
            Object.keys(nextProps).filter(
                key => key !== 'scroll' && this.props[key] !== nextProps[key],
            ).length > 0
        );
    }

    onScroll = e => {
        const scroll = calcRelativeScroll(e.target.scrollTop, this._sizeCache);

        this.props.onScroll(e, {scroll});
    };

    scrollTo(scroll) {
        const DELTA = 1;
        const scrollTop = calcAbsoluteScroll(scroll, this._sizeCache);

        if (Math.abs(this._container.scrollTop - scrollTop) > DELTA) {
            this._container.scrollTop = scrollTop;
        }
    }

    render() {
        const {light, months, MonthComponent, ...props} = this.props;
        const isRedesigned = this.props.isRedesigned;

        delete props.scroll; // скролл не нужно передавать в дочерние компоненты

        return (
            <div className={b({light, redesigned: isRedesigned})}>
                <div
                    className={b('months')}
                    onScroll={this.onScroll}
                    ref={refCallback(this, '_container')}
                >
                    {months.map(month => (
                        <MonthComponent
                            key={month.format(ROBOT_MONTH)}
                            month={month}
                            {...props}
                        />
                    ))}
                </div>

                <div className={b('header')}>
                    {isRedesigned && <CalendarToolbar {...props} />}

                    <Weekdays className={b('weekdays')} {...props} />
                </div>
            </div>
        );
    }
}

CalendarContent.defaultProps = {
    MonthComponent: Month,
    WeekdayComponent: Weekday,
};

CalendarContent.propTypes = {
    months: propTypes.arrayOf(propTypes.moment).isRequired,
    MonthComponent: propTypes.func,
    WeekdayComponent: propTypes.func,
};

export default CalendarContent;
