import {React, B, PropTypes} from '../base';

import {ROBOT} from '../../lib/date/formats';

import {isWeekend} from '../../lib/date/utils';
import noop from '../../lib/noop';

import CalendarDayBase from './CalendarDayBase';

import keyset from '../../i18n/search-hints';

const b = B('CalendarDay');

export default class CalendarDay extends CalendarDayBase {
    static propTypes = {
        day: PropTypes.moment.isRequired,
        today: PropTypes.moment.isRequired,
        range: PropTypes.momentRange.isRequired,
        value: PropTypes.datePickerValue.isRequired,
        onClick: PropTypes.func.isRequired,
        language: PropTypes.string.isRequired,

        onMouseOut: PropTypes.func,
        onMouseOver: PropTypes.func,
    };

    static defaultProps = {
        onMouseOut: noop,
        onMouseOver: noop,
    };

    shouldComponentUpdate(nextProps) {
        return (
            this.isToday(this.props) !== this.isToday(nextProps) ||
            this.isSelected(this.props) !== this.isSelected(nextProps) ||
            this.isInactive(this.props) !== this.isInactive(nextProps)
        );
    }

    render() {
        const {props} = this;
        const {day, onClick, onMouseOut, onMouseOver} = props;

        const date = day.date();
        const weekend = isWeekend(day.weekday());

        if (this.isInactive(props)) {
            return <span className={b({inactive: true, weekend})}>{date}</span>;
        }

        const today = this.isToday(props);
        const mods = {
            today,
            weekend,
            selected: this.isSelected(props),
            outOfMask: this.isOutOfMask(props),
        };

        const formattedDay = day.format(ROBOT);

        return (
            <span
                className={b(mods)}
                data-date={formattedDay}
                onClick={onClick}
                onMouseOut={onMouseOut}
                onMouseOver={onMouseOver}
            >
                {today ? (
                    <span data-date={formattedDay}>
                        <span data-date={formattedDay}>{date}</span>

                        <span
                            className={b('todayTitle')}
                            data-date={formattedDay}
                        >
                            {keyset('today')}
                        </span>
                    </span>
                ) : (
                    date
                )}
            </span>
        );
    }
}
