import React, {PureComponent} from 'react';
import {Moment} from 'moment';
import B from 'bem-cn-lite';

import {refCallback} from '../base';

import once from '../../lib/once';
import {YEAR, HUMAN_MONTH, ROBOT_MONTH} from '../../lib/date/formats';

interface ICalendarMenuProps {
    scroll: number;
    months: Moment[];
    onScroll(e: React.MouseEvent, data: {scroll: number}): void;

    isRedesigned?: boolean;
    inactiveMonths?: Record<string, boolean>;
}

const b = B('CalendarMenu');

export default class CalendarMenu extends PureComponent<ICalendarMenuProps> {
    private _domNode?: HTMLElement;
    private _height?: number;
    private _capture?: {
        Y: number;
        scroll: number;
    };

    componentDidMount(): void {
        if (this._domNode) {
            this._height = this._domNode.clientHeight;
        }
    }

    onClick = (e): void => {
        const scroll = e.target.dataset.index;

        this.props.onScroll(e, {scroll});
    };

    onMouseDown = (e): void => {
        this._capture = {
            Y: e.clientY,
            scroll: this.props.scroll,
        };
        document.addEventListener('mousemove', this.onMouseMove);
        once(document, 'mouseup', this.onMouseUp);
    };

    onMouseMove = (e): void => {
        if (!this._capture || !this._height) {
            return;
        }

        const monthsCount = this.props.months.length;
        let scroll =
            this._capture.scroll +
            ((e.clientY - this._capture.Y) * monthsCount) / this._height;

        scroll = Math.min(Math.max(0, scroll), monthsCount - 2);
        this.props.onScroll(e, {scroll});
    };

    onMouseUp = (): void => {
        document.removeEventListener('mousemove', this.onMouseMove);
    };

    render(): React.ReactNode {
        const {
            months,
            scroll,

            isRedesigned,
            inactiveMonths = {},
        } = this.props;

        const monthHeight = 100 / months.length;
        let prevYear;

        return (
            <div
                className={b({redesigned: isRedesigned})}
                ref={refCallback(this, '_domNode')}
            >
                <div
                    className={b('scroller')}
                    onMouseDown={this.onMouseDown}
                    style={{
                        height: `${monthHeight * (isRedesigned ? 1 : 2)}%`,
                        top: `${monthHeight * scroll}%`,
                    }}
                />
                {months.map((month, index) => {
                    const formattedMonth = month.format(ROBOT_MONTH);
                    const isNextYear = prevYear && prevYear !== month.year();

                    prevYear = month.year();

                    return (
                        <div
                            className={b('month', {
                                inactive: inactiveMonths[formattedMonth],
                            })}
                            key={formattedMonth}
                            onClick={this.onClick}
                            data-index={index}
                        >
                            {month.format(HUMAN_MONTH)}
                            {isNextYear ? (
                                <span className={b('monthYear')}>
                                    {month.format(YEAR)}
                                </span>
                            ) : null}
                        </div>
                    );
                })}
            </div>
        );
    }
}
