import React, {ComponentType, SyntheticEvent, PureComponent} from 'react';
import {Moment} from 'moment';

import {MIN_MONTHS_TO_FULL_CALENDAR} from '../../lib/date/values';

import IMergedMaskDaysOfWalking from '../../interfaces/maskDaysOfWalking/IMergedMaskDaysOfWalking';
import ITime from '../../interfaces/state/searchForm/ITime';
import IWhen from '../../interfaces/state/IWhen';
import IRange from '../../interfaces/date/IRange';
import Lang from '../../interfaces/Lang';

import CalendarMenu from '../CalendarMenu/CalendarMenu';
import CalendarContent from '../CalendarContent/CalendarContent';
import {IWithClassName} from 'common/interfaces/components/IWithClassName';

interface IDayComponentProps {
    day: Moment;
    today: Moment;
    dateStart: Moment;
    range: IRange;
    onClick: (e: SyntheticEvent) => void;
}

interface IWeekdayComponentProps extends IWithClassName {
    weekday: number;
    language: Lang;
}

interface IMonthComponentProps extends IWithClassName {
    month: Moment;
    language: Lang;

    DayComponent?: ComponentType<IDayComponentProps>;
    showMonthName: boolean;
    showYear: boolean;
    showWeekdays: boolean;
    onClick: (e: SyntheticEvent) => void;
}

interface ICalendarScrollerProps<
    M = IMonthComponentProps,
    W = IWeekdayComponentProps,
    D = IDayComponentProps,
> {
    months: Moment[];
    today: Moment;
    language: Lang;
    isRedesigned?: boolean;

    onClick?: (e: SyntheticEvent<HTMLElement>) => void;
    value?: IWhen;
    onMouseOut?: (e: SyntheticEvent) => void;
    onMouseOver?: (e: SyntheticEvent) => void;
    range?: IRange;
    MonthComponent?: ComponentType<M>;
    WeekdayComponent?: ComponentType<W>;
    DayComponent?: ComponentType<D>;
    time?: ITime;
    inactiveMonths?: Record<string, boolean>;
    mask?: IMergedMaskDaysOfWalking;
    nowMoment?: Moment;
}

interface ICalenderScrollerState {
    scroll: number;
}

class CalendarScroller<
    M = IMonthComponentProps,
    W = IWeekdayComponentProps,
    D = IDayComponentProps,
> extends PureComponent<
    ICalendarScrollerProps<M, W, D>,
    ICalenderScrollerState
> {
    state = {scroll: 1};

    onScroll = (e: SyntheticEvent, data: ICalenderScrollerState): void => {
        this.setState(data);
    };

    render(): React.ReactElement {
        const {months, inactiveMonths, isRedesigned} = this.props;
        const {scroll} = this.state;

        const light = months.length < MIN_MONTHS_TO_FULL_CALENDAR;

        return (
            <div className="CalendarScroller">
                {!light && (
                    <CalendarMenu
                        months={months}
                        isRedesigned={isRedesigned}
                        inactiveMonths={inactiveMonths}
                        scroll={scroll}
                        onScroll={this.onScroll}
                    />
                )}
                <CalendarContent
                    {...this.props}
                    light={light}
                    scroll={scroll}
                    redesigned={isRedesigned}
                    onScroll={this.onScroll}
                />
            </div>
        );
    }
}

export default CalendarScroller;
