import {React, PureComponent, B, mix} from '../base';

import Arrow from '../Arrow/Arrow';
import Button from '../Button/Button';
import Checkbox from '../Checkbox/Checkbox';

import componentsKeyset from '../../i18n/components';

const b = B('CheckList');

const ARROW = <Arrow className={b('arrow')} direction="down" />;

export default class CheckList extends PureComponent {
    static defaultProps = {
        showCut: false,
        cutLimit: 5,
    };

    state = {
        expanded: false,
    };

    onChange = e => {
        const {target} = e;
        const {value, options, onChange} = this.props;

        let newValue;

        if (target.checked) {
            newValue = options
                .map(option => option.value)
                .filter(
                    valueItem =>
                        valueItem === target.value || value.includes(valueItem),
                );
        } else {
            newValue = value.filter(valueItem => valueItem !== target.value);
        }

        onChange(e, {
            value: newValue,
        });
    };

    onCutClick = () => {
        this.setState({
            expanded: true,
        });
    };

    isOptionInactive(option) {
        const {activeOptions} = this.props;

        if (!activeOptions) {
            return false;
        }

        return !activeOptions.includes(option.value);
    }

    getColumns(columnsCount) {
        const {options = []} = this.props;
        const columns = [];

        const itemsInColumn = Math.ceil(options.length / columnsCount);

        for (let i = 0; i < columnsCount; i++) {
            const items = options.slice(
                itemsInColumn * i,
                itemsInColumn * (i + 1),
            );

            columns.push(items);
        }

        return columns;
    }

    render() {
        const {
            className,
            value,
            options,
            showCut,
            cutLimit,
            columnsCount = 1,
        } = this.props;
        const {expanded} = this.state;

        let columns = this.getColumns(columnsCount);
        const cutIsAvailable =
            showCut &&
            !expanded &&
            Math.max(...columns.map(column => column.length)) > cutLimit;

        if (cutIsAvailable) {
            columns = columns.map(column => column.slice(0, cutLimit));
        }

        return (
            <div className={mix(b({columned: columnsCount > 1}), className)}>
                {columns.map((column, columnIndex) => (
                    <ul key={columnIndex} className={b('column')}>
                        {column.map(option => (
                            <li
                                className={b('item', {
                                    inactive: this.isOptionInactive(option),
                                    [option.value]: true,
                                })}
                                key={option.value}
                                ref={option.value}
                            >
                                <Checkbox
                                    className={b('checkbox')}
                                    {...option}
                                    checked={value.includes(option.value)}
                                    onChange={this.onChange}
                                />
                            </li>
                        ))}
                    </ul>
                ))}

                {cutIsAvailable && (
                    <Button
                        className={b('cut')}
                        type="button"
                        rightIcon={ARROW}
                        onClick={this.onCutClick}
                    >
                        {componentsKeyset('expand-count-list', {
                            count: options.length - columnsCount * cutLimit,
                        })}
                    </Button>
                )}
            </div>
        );
    }
}
