import {B, PureComponent, React} from '../base';

import {momentTimezone as moment} from '../../../reexports';
import {SyntheticEvent} from 'react';

import {HUMAN, HUMAN_SHORT, ROBOT} from '../../lib/date/formats';

import IconGlyph from '../../interfaces/components/IconGlyph';
import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
import DateRobot from '../../interfaces/date/DateRobot';

import {makeCacheable} from '../../lib/cache';
import getValueFromEnum from '../../lib/enum/getValueFromEnum';

import Icon from '../Icon/Icon';
import Button2 from '../Button2/Button2';

import searchFormKeyset from '../../i18n/search-form';

const b = B('DateButtons');

const getDateSpecialCallback = makeCacheable(
    (
            callback: (special: DateSpecialValue) => void,
            special: DateSpecialValue,
        ) =>
        () =>
            callback(special),
);

interface IDateButtonsProps {
    specialButtons: DateSpecialValue[];
    onSpecialDateClick(special: DateSpecialValue): void;
    onDatepickerClick(e: SyntheticEvent): void;

    whenSpecial?: DateSpecialValue;
    whenDate?: DateRobot;
    calendarButtonHasText?: boolean;
    className?: string;
}

export default class DateButtons extends PureComponent<IDateButtonsProps, {}> {
    static defaultProps = {
        calendarButtonHasText: true,
    };

    isCalendarTabSelected(): boolean {
        const {whenSpecial, specialButtons} = this.props;

        return (
            whenSpecial === undefined ||
            Boolean(
                !specialButtons.includes(whenSpecial) &&
                    getValueFromEnum(whenSpecial, DateSpecialValue),
            )
        );
    }

    getSpecialButtons(): React.ReactElement[] {
        const {specialButtons, onSpecialDateClick, whenSpecial} = this.props;

        return specialButtons.map((special, index) => (
            <Button2
                key={special}
                className={b('dateButton', {
                    selected: whenSpecial === special,
                    unclickableWhenSelected: true,
                })}
                onClick={getDateSpecialCallback(onSpecialDateClick, special)}
                selected={whenSpecial === special}
                borderRadius={index === 0 ? 'left' : 'none'}
                rightSideButtonDelimiter
            >
                {searchFormKeyset(special)}
            </Button2>
        ));
    }

    getCalendarButton(): React.ReactElement {
        const {
            whenSpecial,
            whenDate,
            onDatepickerClick,
            specialButtons,
            calendarButtonHasText,
        } = this.props;

        const calendarTabSelected = this.isCalendarTabSelected();
        const specialIsSelectedFromDatepicker =
            calendarTabSelected &&
            whenSpecial &&
            !specialButtons.includes(whenSpecial);

        if (!calendarTabSelected && !calendarButtonHasText) {
            return (
                <Button2
                    className={b('dateButton', {
                        selected: false,
                        calendarButtonWithoutText: true,
                    })}
                    onClick={onDatepickerClick}
                    selected={false}
                    borderRadius="right"
                >
                    <Icon
                        glyph={IconGlyph.calendar}
                        className={b('calendarIcon')}
                    />
                </Button2>
            );
        }

        const dateAsMoment = moment(whenDate, ROBOT);
        const formatted = dateAsMoment.format(HUMAN).toLowerCase();
        const shortFormatted = dateAsMoment.format(HUMAN_SHORT).toLowerCase();

        return (
            <Button2
                className={b('dateButton', {
                    selected: calendarTabSelected,
                })}
                iconRight={
                    <Icon
                        glyph={IconGlyph.calendar}
                        className={b('calendarIcon')}
                    />
                }
                onClick={onDatepickerClick}
                selected={calendarTabSelected}
                borderRadius="right"
                textAlign="center"
            >
                {calendarTabSelected ? (
                    <span className={b('calendarButtonText')}>
                        <span
                            className={b('calendarButtonLongText')}
                            data-nosnippet
                        >
                            {specialIsSelectedFromDatepicker && whenSpecial
                                ? searchFormKeyset(whenSpecial)
                                : formatted}
                        </span>
                        <span
                            className={b('calendarButtonShortText')}
                            data-nosnippet
                        >
                            {specialIsSelectedFromDatepicker
                                ? searchFormKeyset(`${whenSpecial}-short`)
                                : shortFormatted}
                        </span>
                    </span>
                ) : (
                    <span className={b('calendarButtonText')}>
                        {searchFormKeyset('date')}
                    </span>
                )}
            </Button2>
        );
    }

    render(): React.ReactElement {
        return (
            <div className={b()}>
                {this.getSpecialButtons()}
                {this.getCalendarButton()}
            </div>
        );
    }
}
