import {React, PropTypes, PureComponent, B} from '../base';

import {connect} from 'react-redux';

import {
    HUMAN_SHORT,
    HUMAN_DATE_SHORT_WITH_SHORT_WEEKDAY,
    ROBOT,
} from '../../lib/date/formats';
import {SUBURBAN_TYPE, ALL_TYPE} from '../../lib/transportType';

import DateSpecialValue from '../../interfaces/date/DateSpecialValue';

import searchUrl from '../../lib/url/searchUrl';
import {buildDate} from '../../lib/date/build';
import {getToday} from '../../lib/date/utils';
import {reachGoal} from '../../lib/yaMetrika';
import isAllDaysSearch from '../../lib/search/isAllDaysSearch';

import Link from '../Link';

import calendarKeyset from '../../i18n/calendar';
import suburbanScheduleKeyset from '../../i18n/suburban-schedule';

const b = B('DateHintsLinks');

const mapStateToProps = ({tld, language}) => ({
    tld,
    language,
});

class DateHintsLinks extends PureComponent {
    static propTypes = {
        context: PropTypes.object.isRequired,
        sort: PropTypes.object.isRequired,
        time: PropTypes.object.isRequired,
        currencies: PropTypes.object.isRequired,
    };

    onClick() {
        reachGoal('date_hints_links_click');
    }

    getUrl(params = {}) {
        const {currencies, context, sort, filtering, tld, language} =
            this.props;

        return searchUrl(
            {
                context: {
                    ...context,
                    searchNext: false,
                    ...params,
                },
                sort,
                filtering,
                currencies,
            },
            tld,
            language,
        );
    }

    getDay(date) {
        const {tld, time, context, language, isRedesigned} = this.props;
        const parseParams = {tld, time, language};

        const when = buildDate(date, parseParams);

        return {
            title: date.format(
                isRedesigned
                    ? HUMAN_DATE_SHORT_WITH_SHORT_WEEKDAY
                    : HUMAN_SHORT,
            ),
            link: this.getUrl({when, plan: null}),
            selected: date.format(ROBOT) === context.when.date,
            dataNoSnippet: true,
        };
    }

    getAllDaysTitle({transportType, isNextPlan, plan} = {}) {
        if (plan && transportType === SUBURBAN_TYPE) {
            return plan.title;
        }

        if (plan && isNextPlan) {
            return suburbanScheduleKeyset('all-days-title', {
                title: plan.title,
            });
        }

        return calendarKeyset('all-days-short');
    }

    getHint(day) {
        return day.hint && <span className={b('linkHint')}>{day.hint}</span>;
    }

    render() {
        const {time, plans, context, isRedesigned} = this.props;
        const {when, transportType} = context;
        const allDays = isAllDaysSearch(context);

        const currentDay = getToday(
            allDays
                ? time
                : {
                      now: when.date,
                      timezone: time.timezone,
                  },
        );

        const days = [
            this.getDay(currentDay),
            this.getDay(currentDay.clone().add(1, 'day')),
        ];

        if (!allDays) {
            days.unshift(this.getDay(currentDay.clone().subtract(1, 'day')));
        }

        // Случай смены графиков в электричках по СНГ, обычно в декабре
        if (
            plans.current &&
            plans.next &&
            allDays &&
            (transportType === SUBURBAN_TYPE || transportType === ALL_TYPE)
        ) {
            days.push({
                title: this.getAllDaysTitle({
                    transportType,
                    plan: plans.current,
                }),
                link: this.getUrl({
                    when: {
                        text: calendarKeyset('all-days'),
                        special: DateSpecialValue.allDays,
                    },
                    plan: null,
                }),
                selected: !context.plan || context.plan === plans.current.code,
            });
            days.push({
                title: this.getAllDaysTitle({
                    transportType,
                    isNextPlan: true,
                    plan: plans.next,
                }),
                link: this.getUrl({
                    when: {
                        text: calendarKeyset('all-days'),
                        special: DateSpecialValue.allDays,
                    },
                    plan: plans.next.code,
                    transportType: SUBURBAN_TYPE,
                }),
                hint: calendarKeyset('new-plan'),
                selected: context.plan === plans.next.code,
            });
        } else {
            days.push({
                title: this.getAllDaysTitle(),
                link: this.getUrl({
                    when: {
                        text: calendarKeyset('all-days'),
                        special: DateSpecialValue.allDays,
                    },
                    plan: null,
                }),
                selected: allDays,
            });
        }

        return (
            <div className={b({redesigned: isRedesigned})}>
                {days.map((day, index) =>
                    day.selected ? (
                        <span
                            key={index}
                            className={b('selectedDate')}
                            data-nosnippet={
                                day.dataNoSnippet ? true : undefined
                            }
                        >
                            {day.title}
                            {this.getHint(day)}
                        </span>
                    ) : (
                        <Link
                            key={index}
                            className={b('link')}
                            href={day.link}
                            onClick={this.onClick}
                            data-nosnippet={
                                day.dataNoSnippet ? true : undefined
                            }
                        >
                            {day.title}
                            {this.getHint(day)}
                        </Link>
                    ),
                )}
            </div>
        );
    }
}

export default connect(mapStateToProps)(DateHintsLinks);
