import {refCallback} from '../base';

import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';

import KeyCode from '../../interfaces/KeyCode';
import IconGlyph from '../../interfaces/components/IconGlyph';

import parseDate from '../../lib/date/parse';
import {getParseParams} from '../../lib/date/utils';

import Icon from '../Icon/Icon';

const b = B('DateInput');

export default class DateInput extends PureComponent {
    state = {value: this.props.value};

    componentWillReceiveProps({value}) {
        this.setState({value});
    }

    onKeyPress = e => {
        if (e.keyCode === KeyCode.enter) {
            if (this.applyCurrentValue(e)) {
                e.preventDefault();
            } else {
                this.triggerLeaveFocus();
            }
        }
    };

    onChange = e => {
        const value = parseDate(e.target.value, getParseParams(this.props));

        this.setValue(e, value);
    };

    onBlur = e => {
        this.applyCurrentValue(e);
        this.props.onBlur();
    };

    triggerLeaveFocus() {
        setTimeout(() => {
            this._input.blur();
        });
    }

    applyCurrentValue(e) {
        const {value} = this.props;
        const {text, formatted} = value;

        if (formatted && text !== formatted) {
            this.setValue(e, {
                ...value,
                hint: formatted,
                text: formatted,
            });

            return true;
        }
    }

    setValue(e, value) {
        const data = {value};

        this.setState(data);
        this.props.onChange(e, data);
    }

    render() {
        const {value} = this.state;
        const {
            id,
            name,
            placeholder,
            onFocus,
            hint,
            defaultValue,
            autoFocus,
            onClick,
        } = this.props;
        const inputValue =
            value.text && value.text.trim() ? value.text : defaultValue;
        const hintText = hint || value.hint;

        return (
            <div className={b()}>
                <div
                    className={b('hint', {onTop: hint && hint !== value.hint})}
                >
                    {hintText !== value.text && hintText}
                </div>

                <input
                    className={b('input')}
                    id={id}
                    type="text"
                    ref={refCallback(this, '_input')}
                    value={value.text}
                    onBlur={this.onBlur}
                    onFocus={onFocus}
                    onChange={this.onChange}
                    onKeyDown={this.onKeyPress}
                    autoComplete="off"
                    placeholder={placeholder}
                    autoFocus={autoFocus}
                    onClick={onClick}
                    data-nosnippet
                />

                <input
                    type="hidden"
                    name={name}
                    value={inputValue}
                    data-nosnippet
                />

                <label className={b('icon')} htmlFor={id}>
                    <Icon className={b('iconSvg')} glyph={IconGlyph.calendar} />
                </label>
            </div>
        );
    }
}
