import React, {PureComponent} from 'react';
import throttle from 'lodash/throttle';
import B from 'bem-cn-lite';

import {ANIMATION_DURATION} from '../Popup/Popup';

import SecondaryPosition from '../../interfaces/lib/dimensions/SecondaryPosition';
import PrimaryPosition from '../../interfaces/lib/dimensions/PrimaryPosition';

// eslint-disable-next-line no-duplicate-imports
import Popup from '../Popup/Popup';
import DateInput from '../DateInput/DateInput';
import SearchFormCalendar from '../SearchFormCalendar/SearchFormCalendar';

const POSITIONS = [
    [PrimaryPosition.bottom, SecondaryPosition.left],
    [PrimaryPosition.bottom, SecondaryPosition.right],
    [PrimaryPosition.left, SecondaryPosition.top],
];

const REDESIGNED_POSITIONS = [
    [PrimaryPosition.bottom, SecondaryPosition.center],
];

const b = B('DatePicker');

// На Windows у thinkpad ноутбуков, при тапе на тачпад, фокус полей происходит с задержкой.
const TOUCHPAD_DELAY = 100;

class DatePicker extends PureComponent {
    state = {
        hint: '',
        calendarVisible: this.props.errors.length > 0,
    };

    constructor(...args) {
        super(...args);

        // Такие заморочки из-за бага в сафари
        // https://github.com/facebook/react/issues/10871
        this.setCalendarVisibilityThrottled = throttle(
            this.setCalendarVisibility,
            ANIMATION_DURATION + TOUCHPAD_DELAY,
            {leading: true, trailing: false},
        );
    }

    componentWillReceiveProps(props) {
        if (props.errors.length > 0) {
            this.setCalendarVisibility(true);
        }
    }

    onFocus = () => {
        if (!this.state.calendarVisible) {
            this.setCalendarVisibilityThrottled(true);
        }
    };

    onBlur = () => {
        if (this.state.calendarVisible) {
            this.setCalendarVisibilityThrottled(false);
        }
    };

    onPopupClickOutside = e => {
        if (!this._self.contains(e.target) && this.state.calendarVisible) {
            this.setCalendarVisibilityThrottled(false);
        }
    };

    onInputChange = (e, data) => {
        this.emptyHint();
        this.props.onChange(e, data);
    };

    onCalendarClick = (e, data) => {
        this.props.onChange(e, data);

        if (this.state.calendarVisible) {
            this.setCalendarVisibilityThrottled(false);
        }
    };

    onCalendarMouseOver = (e, data) => {
        this.setState(data);
    };

    onCalendarMouseOut = () => {
        this.emptyHint();
    };

    onClick = () => {
        if (!this.state.calendarVisible) {
            this.setCalendarVisibilityThrottled(true);
        }
    };

    emptyHint() {
        this.setState({hint: ''});
    }

    makeSelfRef = node => {
        this._self = node;
    };

    setCalendarVisibility = calendarVisible => {
        this.setState({calendarVisible});
    };

    render() {
        const props = this.props;
        const isRedesigned = props.isRedesigned;
        const {hint, calendarVisible} = this.state;

        return (
            <div
                className={b({redesigned: isRedesigned})}
                ref={this.makeSelfRef}
            >
                <DateInput
                    {...props}
                    hint={hint}
                    onBlur={this.onBlur}
                    onFocus={this.onFocus}
                    onChange={this.onInputChange}
                    onClick={this.onClick}
                />
                <Popup
                    className={b('popup')}
                    arrowClassName={b('popupArrow')}
                    contentClassName={b('popupContent')}
                    visible={calendarVisible}
                    withoutArrow
                    positions={isRedesigned ? REDESIGNED_POSITIONS : POSITIONS}
                    theme={isRedesigned ? 'travel' : undefined}
                    onClickOutside={this.onPopupClickOutside}
                >
                    <SearchFormCalendar
                        {...props}
                        onClick={this.onCalendarClick}
                        onMouseOut={this.onCalendarMouseOut}
                        onMouseOver={this.onCalendarMouseOver}
                    />
                </Popup>
            </div>
        );
    }
}

export default DatePicker;
