import {PropTypes} from '../base';

import React, {Component, createRef} from 'react';
import B from 'bem-cn-lite';
import {connect} from 'react-redux';

import noop from '../../lib/noop';

const b = B('Direct');

let pageCounter = 1;

const EVENT_CLOSE = 'ya-internal-close';

const mapStateToProps = ({nonce, page}) => ({nonce, page});

class Direct extends Component {
    static getPageFetching(page) {
        return (page && page.fetching) || null;
    }

    static getPageCurrent(page) {
        return page?.current || null;
    }

    static makeContainerId(state) {
        return `banner-${state.blockId}-${state.pageNumber}`;
    }

    _onCloseListeners = [];
    block = createRef();

    static propTypes = {
        blockId: PropTypes.string.isRequired,

        page: PropTypes.object, // объект страницы из state (свойство page)
        style: PropTypes.object,
        className: PropTypes.string,
        grab: PropTypes.string,
        format: PropTypes.string,
        position: PropTypes.string,
        onRender: PropTypes.func, // будет вызвана, когда рекламный блок отрендерится
        onNotRender: PropTypes.func, // будет вызвана в том случае, если баннер был запрошен,
        // но система директа ничего не вернула, т.е. не подобрала нужный баннер для страницы
        onClose: PropTypes.func, // будет вызван, если пользователь закрыл баннер
        updateIfLoadedAnotherPage: PropTypes.bool, // нужно ли загружать новый баннер при обновлении страницы
    };

    static defaultProps = {
        className: '',
        grab: '',
        format: '',
        position: '',
        onRender: noop,
        onNotRender: noop,
        onClose: noop,
        page: {},
        style: {},
        updateIfLoadedAnotherPage: true,
    };

    state = {
        blockId: this.props.blockId,
        blockIdChangedDuringFetching: false,
        pageFetching: Direct.getPageFetching(this.props.page),
        pageFetchingWhenCreate: Direct.getPageFetching(this.props.page),
        pageCurrent: Direct.getPageCurrent(this.props.page),
        countChangePageFetching: 0,
        pageNumber: null,
        requestingBlockId: null,
        mounted: false,
    };

    static getDerivedStateFromProps(props, state) {
        const newState = {};

        if (props.blockId !== state.blockId) {
            newState.blockId = props.blockId;
            newState.blockIdChangedDuringFetching = true;
        }

        const pageFetching = Direct.getPageFetching(props.page);
        const pageCurrent = Direct.getPageCurrent(props.page);

        if (pageFetching !== state.pageFetching) {
            newState.pageFetching = pageFetching;
            newState.countChangePageFetching =
                state.countChangePageFetching + 1;
        }

        if (
            props.updateIfLoadedAnotherPage &&
            pageFetching !== state.pageFetching &&
            pageFetching === null &&
            // обработка для случая, когда на клиенте при переходе на другую страницу
            // компонент монтируется, а потом происходит событие `finishFetchingPage`
            // и меняется page.fetching и компонент думает, что произошел переход на другую
            // страницу и заново запрашивает рекламу
            !(
                (state.pageFetching === state.pageFetchingWhenCreate &&
                    newState.countChangePageFetching === 1) ||
                // вариант на случай если компонент всегда есть на странице - приходится ориентироваться
                // по тому что произошёл переход на другую страницу с блоком с другим id'шником и увеличивать
                // номер страницы не нужно
                (state.pageCurrent !== pageCurrent &&
                    state.blockIdChangedDuringFetching &&
                    newState.countChangePageFetching === 2)
            )
        ) {
            newState.pageNumber = pageCounter++;
        }

        if (pageFetching === null) {
            newState.pageCurrent = pageCurrent;
            newState.countChangePageFetching = 0;
            newState.blockIdChangedDuringFetching = false;
        }

        return Object.keys(newState).length ? newState : null;
    }

    componentDidMount() {
        this.setState({
            mounted: true,
            pageNumber: pageCounter++,
        });
    }

    shouldComponentUpdate(nextProps, nextState) {
        return (
            Direct.makeContainerId(this.state) !==
            Direct.makeContainerId(nextState)
        );
    }

    componentDidUpdate(prevProps, prevState) {
        if (
            Direct.makeContainerId(this.state) !==
            Direct.makeContainerId(prevState)
        ) {
            this.renderDirect();
        }
    }

    componentWillUnmount() {
        this.removeOnCloseListeners();
    }

    addOnCloseListener(listener) {
        if (!this.domElement) {
            return;
        }

        this.domElement.addEventListener(EVENT_CLOSE, listener);
        this._onCloseListeners.push(listener);

        return this._onCloseListeners;
    }

    removeOnCloseListeners() {
        if (!this.domElement) {
            return;
        }

        this._onCloseListeners.forEach(listener => {
            this.domElement.removeEventListener(EVENT_CLOSE, listener);
        });
        this._onCloseListeners = [];

        return this._onCloseListeners;
    }

    getContainerId() {
        return Direct.makeContainerId(this.state);
    }

    renderDirect = () => {
        const {grab, format, position, nonce, onRender, onNotRender, onClose} =
            this.props;
        const {pageNumber, blockId, requestingBlockId} = this.state;
        const containerId = this.getContainerId();
        const dataForCallbacks = {
            elementId: containerId,
            blockId: this.state.blockId,
            ref: this.block,
        };

        if (requestingBlockId !== blockId) {
            this.setState({
                requestingBlockId: blockId,
            });

            window.yaContextCb.push(() => {
                Ya.Context.AdvManager.render(
                    {
                        blockId,
                        ...(grab ? {grab} : null),
                        ...(format ? {format} : null),
                        ...(position ? {position} : null),
                        pageNumber,
                        renderTo: containerId,
                        async: true,
                        cspNonce: nonce,
                        onRender: (...args) => {
                            this.setState({
                                requestingBlockId: null,
                            });

                            onRender(dataForCallbacks, ...args);

                            this.addOnCloseListener(e =>
                                onClose(dataForCallbacks, e),
                            );
                        },
                    },
                    () => onNotRender(dataForCallbacks),
                );
            });
        }
    };

    render() {
        const {className, style} = this.props;
        const {mounted} = this.state;
        const params = {};

        if (mounted) {
            params.id = this.getContainerId();
        }

        return (
            <div
                className={b(null, className)}
                style={style}
                ref={this.block}
                {...params}
            />
        );
    }
}

export default connect(mapStateToProps)(Direct);
