import {refCallback} from '../base';

import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';

import KeyCode from '../../interfaces/KeyCode';
import SecondaryPosition from '../../interfaces/lib/dimensions/SecondaryPosition';
import PrimaryPosition from '../../interfaces/lib/dimensions/PrimaryPosition';

import noop from '../../lib/noop';

import Arrow from '../Arrow/Arrow';
import Popup from '../Popup/Popup';
import Button from '../Button/Button';

const b = B('Dropdown');
const POPUP_POSITIONS = [
    [PrimaryPosition.bottom, SecondaryPosition.left],
    [PrimaryPosition.bottom, SecondaryPosition.center],
    [PrimaryPosition.bottom, SecondaryPosition.right],
];

export default class Dropdown extends PureComponent {
    state = {
        opened: false,
    };

    componentWillUnmount() {
        this.removeListeners();
    }

    onButtonClick = e => {
        const {opened} = this.state;

        if (opened) {
            this.close(e);
        } else {
            this.open(e);
        }
    };

    onDocumentClick = e => {
        const {closeAfterClick} = this.props;

        if (!this._self.contains(e.target) || closeAfterClick) {
            this.close(e);
        }
    };

    onDocumentKeyDown = e => {
        if (e.keyCode === KeyCode.esc) {
            this.close();
        }
    };

    open(e) {
        if (!this.state.opened) {
            this.setState({opened: true});
            this.props.onOpen(e);
            this.addListeners();
        }
    }

    close(e) {
        if (this.state.opened) {
            this.setState({opened: false});
            this.props.onClose(e);
            this.removeListeners();
        }
    }

    addListeners() {
        document.addEventListener('click', this.onDocumentClick);
        document.addEventListener('keydown', this.onDocumentKeyDown);
    }

    removeListeners() {
        document.removeEventListener('click', this.onDocumentClick);
        document.removeEventListener('keydown', this.onDocumentKeyDown);
    }

    render() {
        const {
            className,
            isMobile,
            children,
            buttonText,
            title,
            inactive,
            highlighted,
            noArrow = false,
            showButtonArrow = true,
            popupPositions,
            getContent,
            size,
        } = this.props;

        const {opened} = this.state;
        const arrow = !noArrow && (
            <Arrow className={b('arrow')} direction={opened ? 'up' : 'down'} />
        );
        const mods = {
            opened,
            noArrow,
            isMobile,
            size,
        };

        return (
            <div
                className={b(mods, className)}
                ref={refCallback(this, '_self')}
            >
                <Button
                    className={b('toggler', {highlighted, inactive})}
                    active={opened}
                    rightIcon={showButtonArrow && arrow}
                    onClick={this.onButtonClick}
                    title={title}
                >
                    {buttonText}
                </Button>

                <Popup
                    className={b('content')}
                    visible={opened}
                    positions={popupPositions || POPUP_POSITIONS}
                    getContent={getContent}
                >
                    {children}
                </Popup>
            </div>
        );
    }
}

Dropdown.defaultProps = {
    onOpen: noop,
    onClose: noop,
    size: 'medium',
};
