import {React, PureComponent, B} from '../base';

import moment from 'moment';

import {HOURS} from '../../lib/date/formats';
import {TIMES_OF_DAY_LIST, TIMES_RANGES} from '../../lib/date/values';
import {reachGoal} from '../../lib/yaMetrika';

import TimeOfDay from '../../interfaces/date/TimeOfDay';

import noop from '../../lib/noop';
import arrivalFilter from '../../lib/filters/arrival';
import departureFilter from '../../lib/filters/departure';

import RadioGroup from '../basic/RadioGroup';
import ButtonCheckbox from '../ButtonCheckbox/ButtonCheckbox';

import keyset from '../../i18n/filter-departure-arrival-time';

const b = B('FilterTimeOfDayContent');
const filterTypes = [departureFilter.type, arrivalFilter.type];
const filterTypeButtons = filterTypes.map(value => ({
    value,
    label: keyset(value),
}));

export default class FilterTimeOfDayContent extends PureComponent {
    static defaultProps = {
        onChange: noop,
    };

    state = {
        eventType: 'departure',
    };

    onClick = {
        [TimeOfDay.night]: () => this.setFilterValue(TimeOfDay.night),
        [TimeOfDay.morning]: () => this.setFilterValue(TimeOfDay.morning),
        [TimeOfDay.day]: () => this.setFilterValue(TimeOfDay.day),
        [TimeOfDay.evening]: () => this.setFilterValue(TimeOfDay.evening),
    };

    componentDidMount() {
        const {onMount} = this.props;

        reachGoal('filter_departure_show');

        if (onMount) {
            onMount();
        }
    }

    onTogglerChange = (e, {value}) => {
        reachGoal(`filter_${value}_show`);

        this.setState({
            eventType: value,
        });
    };

    setFilterValue(timeOfDay) {
        const {eventType} = this.state;
        const {
            onChange,
            [eventType]: {value},
        } = this.props;
        const times = value.filter(item => item !== timeOfDay);

        if (times.length === value.length) {
            times.push(timeOfDay);
        }

        onChange({
            type: eventType,
            value: times,
        });
    }

    render() {
        const {eventType} = this.state;
        const {
            [eventType]: {value, activeOptions},
        } = this.props;
        const timeFormat = keyset('time-format');

        return (
            <div className={b()}>
                <RadioGroup
                    className={b('toggler')}
                    value={eventType}
                    buttons={filterTypeButtons}
                    onChange={this.onTogglerChange}
                />

                {TIMES_OF_DAY_LIST.map(timeOfDay => {
                    return (
                        <ButtonCheckbox
                            key={timeOfDay}
                            className={b('timeOfDay')}
                            value={value.includes(timeOfDay)}
                            active={activeOptions.includes(timeOfDay)}
                            onClick={this.onClick[timeOfDay]}
                        >
                            <span className={b('timeName')}>
                                {keyset(timeOfDay)}
                            </span>

                            <span className={b('hours')}>
                                {keyset('time-range', {
                                    start: moment(
                                        TIMES_RANGES[timeOfDay].start,
                                        HOURS,
                                    ).format(timeFormat),
                                    end: moment(
                                        TIMES_RANGES[timeOfDay].end,
                                        HOURS,
                                    ).format(timeFormat),
                                })}
                            </span>
                        </ButtonCheckbox>
                    );
                })}
            </div>
        );
    }
}
