import React, {useState, useEffect, PureComponent} from 'react';
import B from 'bem-cn-lite';
import {connect, batch} from 'react-redux';

import {DEFAULT_SORT} from '../../lib/sort/utils';
import {FIXED_ELEMENT_CLASS_NAME} from '../_constants/fixedElementClassName';

import IconGlyph from '../../interfaces/components/IconGlyph';

import {reachGoal} from '../../lib/yaMetrika';
import priceFilter from '../../lib/filters/pricePresence';
import {
    setFilterValue,
    filtersWereApplied,
    resetFiltersToDefault,
} from '../../lib/filters/filtering';
import getIsLandscapeView from '../../../client/lib/isLandscapeView';

import {updateSorting} from '../../actions/search/sorting';
import {updateFiltering} from '../../actions/search/filters';

import Icon from '../Icon/Icon';
import Button2 from '../Button2/Button2';
import GlobalPopup from '../GlobalPopup';
import SearchSorting from '../search/SearchSorting/SearchSorting';
import FilterExpress from '../FilterExpress';
import FilterCarriers from '../FilterCarriers/FilterCarriers';
import FilterTransport from '../FilterTransport/FilterTransport';
import FilterHighSpeedTrain from '../FilterHighSpeedTrain/FilterHighSpeedTrain';
import FilterLastochka from '../FilterLastochka/mobile';
import FilterPricePresence from '../FilterPricePresence/FilterPricePresence';
import {
    FilterStationsTo,
    FilterStationsFrom,
} from '../FilterStations/FilterStation';
import FilterAeroexpress from '../FilterAeroexpress/FilterAeroexpress';
import FilterTimeOfDay from '../FilterTimeOfDay';
import FilterTrainTariffClass from '../FilterTrainTariffClass/FilterTrainTariffClass';
import ModalHistoryWrapper from '../ModalHistoryWrapper/ModalHistoryWrapper';

import keyset from '../../i18n/filters';

const b = B('FiltersToolbar');

const FILTERS = {
    transport: FilterTransport,
    express: FilterExpress,
    pricePresence: FilterPricePresence,
    carriers: FilterCarriers,
    lastochka: FilterLastochka,
    highSpeedTrain: FilterHighSpeedTrain,
    stationTo: FilterStationsTo,
    stationFrom: FilterStationsFrom,
    aeroexpress: FilterAeroexpress,
    trainTariffClass: FilterTrainTariffClass,
};

class FiltersToolbar extends PureComponent {
    state = {
        opened: false,
        filtering: this.props.filtering,
        sort: this.props.sort,
        filtersWereUsed: false,
    };

    componentDidUpdate(prevProps) {
        if (prevProps.filtering !== this.props.filtering) {
            this.mergeFilters(this.props.filtering);
        }
    }

    onApply = () => {
        const {dispatch} = this.props;

        reachGoal('search_form_filters-toolbar_apply_click');
        batch(() => {
            dispatch(updateFiltering(this.state.filtering));
            dispatch(updateSorting(this.state.sort));
        });

        this.setState({opened: false});
    };

    onReset = () => {
        const {segments, dispatch} = this.props;
        const filtering = resetFiltersToDefault(this.state.filtering, segments);

        reachGoal('search_form_filters-toolbar_reset_click');
        batch(() => {
            dispatch(updateSorting(DEFAULT_SORT));
            dispatch(updateFiltering(filtering));
        });

        this.setState({
            sort: DEFAULT_SORT,
            filtering,
        });
    };

    onFilterChange = ({type, value}) => {
        const {segments} = this.props;
        const {filtering} = this.state;

        const newFiltering = setFilterValue({
            value,
            segments,
            filtering,
            filterType: type,
        });

        if (!this.state.filtersWereUsed) {
            reachGoal('search_form_filters-toolbar_filter_changed');
        }

        this.setState({
            filtering: newFiltering,
            filtersWereUsed: true,
        });
    };

    onSortingChange = sort => {
        this.setState({sort});

        reachGoal('search_form_filters-toolbar_sorting_changed');
    };

    onFilterButtonClick = () => {
        reachGoal('search_form_filters-toolbar_click');

        this.setState({
            opened: true,
            filtering: this.props.filtering,
        });
    };

    onFilterToolbarCloseClick = () => {
        reachGoal('search_form_filters-toolbar_filter_close_click');
        this.setState({
            opened: false,
            filtersWereUsed: false,
        });
    };

    mergeFilters = nextFiltering => {
        const {segments} = this.props;

        const preparedFiltering = Object.entries(
            this.state.filtering.filters,
        ).reduce((filtering, [filterType, filtersValues]) => {
            if (!filtering.filters.hasOwnProperty(filterType)) {
                return filtering;
            }

            return setFilterValue({
                value: filtersValues.value,
                segments,
                filtering,
                filterType,
            });
        }, nextFiltering);

        this.setState({filtering: preparedFiltering});
    };

    getGlobalPopup() {
        const {transportTypes, language} = this.props;
        const {sort, filtering} = this.state;
        const {filters} = filtering;

        return (
            <ModalHistoryWrapper onClose={this.onFilterToolbarCloseClick}>
                <GlobalPopup>
                    <FiltersContainer>
                        <div className={b('actions', FIXED_ELEMENT_CLASS_NAME)}>
                            <Button2
                                className={b('reset')}
                                onClick={this.onReset}
                                disabled={
                                    !filtersWereApplied(filtering) &&
                                    this.isSortDefault(sort)
                                }
                            >
                                {keyset('reset-filters')}
                            </Button2>
                            <Button2
                                className={b('apply')}
                                onClick={this.onApply}
                            >
                                {keyset('apply-filters')}
                            </Button2>
                        </div>
                        <div
                            className={b('closeButton')}
                            onClick={this.onFilterToolbarCloseClick}
                        >
                            <Icon
                                className={b('backIcon')}
                                glyph={IconGlyph.backArrow}
                            />
                            {keyset('return-button')}
                        </div>
                        {Object.keys(FILTERS).map(key => {
                            const Filter = FILTERS[key];
                            const filterData = filters[key];

                            return this.isFilterAvailable(filterData) ? (
                                <div className={b('item')} key={key}>
                                    <div className={b('filterTitle')}>
                                        {keyset(key)}
                                    </div>
                                    <Filter
                                        className={b('filter')}
                                        {...filterData}
                                        transportTypes={transportTypes}
                                        onChange={this.onFilterChange}
                                        language={language}
                                    />
                                </div>
                            ) : null;
                        })}

                        <div className={b('item')}>
                            <div className={b('filterTitle')}>
                                {keyset('timeOfDay')}
                            </div>
                            <FilterTimeOfDay
                                className={b('filter')}
                                arrival={filters.arrival}
                                departure={filters.departure}
                                onChange={this.onFilterChange}
                            />
                        </div>

                        <div className={b('item')}>
                            <div className={b('filterTitle')}>
                                {keyset('sorting')}
                            </div>

                            <SearchSorting
                                className={b('filter')}
                                sort={sort}
                                onChange={this.onSortingChange}
                                language={language}
                            />
                        </div>
                    </FiltersContainer>
                </GlobalPopup>
            </ModalHistoryWrapper>
        );
    }

    isSortDefault() {
        const {sort} = this.state;

        return (
            sort.by === DEFAULT_SORT.by && sort.reverse === DEFAULT_SORT.reverse
        );
    }

    isFilterAvailable(filter) {
        if (filter) {
            return filter.type === priceFilter.type
                ? this.props.queryingPrices || filter.availableWithOptions
                : filter.availableWithOptions;
        }

        return false;
    }

    render() {
        const {className, filtering, sort} = this.props;
        const {opened} = this.state;
        const hasFilters =
            filtersWereApplied(filtering) || !this.isSortDefault(sort);

        return (
            <div className={b(null, className)}>
                <Button2
                    className={b('button', {hasFilters})}
                    onClick={this.onFilterButtonClick}
                    sizeMod="m"
                    iconLeft={
                        <Icon
                            className={b('filtersIcon')}
                            glyph={IconGlyph.filters}
                        />
                    }
                >
                    {keyset('title')}
                </Button2>

                {opened && this.getGlobalPopup()}
            </div>
        );
    }
}

export default connect()(FiltersToolbar);

// Далее перенос логики из GlobalPopup, которая почему-то оказалась именно там, хотя ей там не место
function FiltersContainer({children}) {
    const [isLandscapeView, setIsLandscapeView] = useState(
        getIsLandscapeView(),
    );

    useEffect(() => {
        function onResize() {
            const newIsLandscapeView = getIsLandscapeView();

            if (newIsLandscapeView !== isLandscapeView) {
                // Фикс бага, который ломает вёрстку после поворота экрана до первого скролла
                window.scrollTo(window.pageXOffset, window.pageYOffset);

                setIsLandscapeView(newIsLandscapeView);
            }
        }

        window.addEventListener('resize', onResize);

        return () => {
            window.removeEventListener('resize', onResize);
        };
    }, [isLandscapeView]);

    return <div className={b('container', {isLandscapeView})}>{children}</div>;
}
