import {React, PureComponent, B} from '../base';

import Cookies from 'cookies-js';

import {ERROR_TYPES} from '../../lib/errors/errorTypes';
import {SEARCH} from '../../routes/search';
import {HOME_PAGE_NAME} from '../../routes';
import {PAGE_404} from '../../routes/page404';
import {TRANSPORT_CITY_PAGE_NAME} from '../../routes/transportCity';
import {TRANSPORT_PAGE_NAME} from '../../routes/transport';
import {STATION_PAGE_NAME} from '../../routes/station';

import {reachGoal} from '../../lib/yaMetrika';

import Personal from './Personal';
import SearchForm from '../SearchForm/SearchForm';
import HeaderLogo from '../HeaderLogo/HeaderLogo';
import TransportSelector from '../TransportSelector/TransportSelector';
import Slogan from './Slogan';
import Error404Block from '../Error404Block';

const b = B('Header');
const PAGES_WITH_FIXED_HEADER = [SEARCH, ERROR_TYPES.AMBIGUOUS];
const PAGES_WITHOUT_FOCUSED_HEADER = [
    HOME_PAGE_NAME,
    PAGE_404,
    TRANSPORT_CITY_PAGE_NAME,
    TRANSPORT_PAGE_NAME,
];

export const HEADER_HEIGHT = 73;

export default class Header extends PureComponent {
    state = {
        focus: false,
    };

    componentDidMount() {
        this.reachHintsGoal();
    }

    componentDidUpdate(prevProps, prevState) {
        if (!prevState.focus) {
            this.reachHintsGoal();
        }

        if (prevProps.page.fullUrl !== this.props.page.fullUrl) {
            this.setState({focus: false});
        }
    }

    onFocus = () => {
        const thinHeaderStatus = Cookies.get('thin_header');

        if (!thinHeaderStatus) {
            this.setState({focus: true});
        }
    };

    onBlur = () => {
        this.setState({focus: false});
    };

    onSubmit = () => {
        this.setState({focus: false});
    };

    getHeightAt() {
        return this.isFixed() ? HEADER_HEIGHT : 0;
    }

    getPopularDirectionsHints(popularDirections) {
        return {
            popularDirections: popularDirections.map(direction => ({
                value: {...direction},
            })),
        };
    }

    reachHintsGoal() {
        if (
            this.state.focus &&
            this.props.searchFormHints.searchHistory.length
        ) {
            reachGoal('search_hints_station_hints_shown');
        }
    }

    isFixed() {
        return PAGES_WITH_FIXED_HEADER.includes(this.props.page.current);
    }

    // При переходе с главной на страницу станции убираем анимацию шапки
    isNoAnimation() {
        const {
            page: {fetching, current, prevPage},
        } = this.props;

        const currentPage = fetching || current;

        if (prevPage === HOME_PAGE_NAME && currentPage === STATION_PAGE_NAME) {
            return true;
        }

        return false;
    }

    getSearchFormHints() {
        const {transport, transportCity, pageType, searchFormHints} =
            this.props;

        switch (pageType) {
            case TRANSPORT_PAGE_NAME:
                return this.getPopularDirectionsHints(
                    transport.popularDirections,
                );
            case TRANSPORT_CITY_PAGE_NAME:
                return this.getPopularDirectionsHints(
                    transportCity.popularDirections,
                );

            default:
                return searchFormHints;
        }
    }

    render() {
        const {
            tld,
            page,
            user,
            dispatch,
            suggests,
            searchForm,
            nationalVersion,
            environment,
            clientSettlement,
            currentSettlement,
            transportCity,
            transport,
            home,
            pageType,
        } = this.props;

        const isHomePage = pageType === HOME_PAGE_NAME;
        const is404Page = pageType === PAGE_404;
        const isTransportCityPage = pageType === TRANSPORT_CITY_PAGE_NAME;
        const isTransportPage = pageType === TRANSPORT_PAGE_NAME;

        const noAnimation = this.isNoAnimation();

        const mods = {
            fixed: this.isFixed(),
            focused:
                !PAGES_WITHOUT_FOCUSED_HEADER.includes(pageType) &&
                this.state.focus,
            noAnimation,
            pageType,
        };

        return (
            <header className={b(mods)}>
                <div className={b('bar')}>
                    <div className={b('container')}>
                        <div className={b('logo')}>
                            <HeaderLogo isFull={isHomePage} tld={tld} />
                        </div>

                        <Personal
                            className={b('personal')}
                            tld={tld}
                            user={user}
                            page={page}
                            clientSettlement={clientSettlement}
                            showTime={!isHomePage && !isTransportCityPage}
                        />

                        {!(isTransportCityPage && page.fetching) && (
                            <Slogan
                                pageType={pageType}
                                tld={tld}
                                page={page}
                                transportCityType={transportCity.transportType}
                                transportType={transport.transportType}
                                currentSettlement={
                                    pageType === TRANSPORT_CITY_PAGE_NAME
                                        ? transportCity.settlement
                                        : currentSettlement
                                }
                                countries={transport.countries}
                                showAppearAnimation={isTransportCityPage}
                                isCity={home.isCity}
                                hidden={
                                    !isTransportCityPage &&
                                    !isTransportPage &&
                                    !isHomePage
                                }
                                noAnimation={noAnimation}
                            />
                        )}

                        {is404Page && <Error404Block />}

                        <TransportSelector
                            value={searchForm.transportType}
                            dispatch={dispatch}
                            noAnimation={noAnimation}
                        />

                        <SearchForm
                            {...searchForm}
                            focused={this.state.focus}
                            searchFormHints={this.getSearchFormHints()}
                            dispatch={dispatch}
                            onSubmit={this.onSubmit}
                            onFocus={this.onFocus}
                            suggests={suggests}
                            nationalVersion={nationalVersion}
                            clientSettlement={clientSettlement}
                            environment={environment}
                        />
                    </div>
                </div>

                <div className={b('overlay')} onClick={this.onBlur} />
            </header>
        );
    }
}
