import {React, B, refCallback, PureComponent} from '../base';

import {HOME_PAGE_NAME} from '../../routes';
import {TRANSPORT_CITY_PAGE_NAME} from '../../routes/transportCity';
import {TRANSPORT_PAGE_NAME} from '../../routes/transport';
import {CHAR_NBSP} from '../../lib/stringUtils';

import AdFoxPosition from '../../interfaces/components/adFox/AdFoxPosition';
import AdFoxType from '../../interfaces/components/adFox/AdFoxType';

import buildSearchFormSuggestDataProviderOptions from '../../lib/suggests/buildSearchFormSuggestDataProviderOptions';
import {
    setFromPointFromUser,
    setToPointFromUser,
    setWhen,
} from '../../actions/searchForm';
import {reachGoal} from '../../lib/yaMetrika';

import RootContext from '../Root/RootContext';
import LogoAndPersonal from './LogoAndPersonal.mobile';
import SearchForm from '../SearchForm/SearchForm';
import Suggest from '../Suggest/Suggest';
import Sticky from '../Sticky/Sticky';
import Teasers from '../HomePage/Teasers/Teasers';
import GlobalPopup from '../GlobalPopup';
import AdFoxBanner from '../AdFoxBanner/AdFoxBanner';

import sloganKeyset from '../../i18n/slogan';
import transportCityKeyset from '../../i18n/transport-city';
import transportKeyset from '../../i18n/transport';

const b = B('Header');

const HEIGHT_HEADER_STICKY = 80;

export default class Header extends PureComponent {
    constructor(props) {
        super(props);
        this.state = {
            suggestField: undefined,
            headerIsSticky: false, // признак того, что форма поиска прилипла к верху экрана
            searchFormPreOpen: false, // состояние, которое предшествует разворачиванию формы поиска (см. метод onOpenSearchForm)
            searchFormPostClose: false, // состояние после сворачивания формы поиска (см. метод onCloseSearchForm)
            searchFormTop: null, // позиция поисковой формы относительно окна браузера до разворачивания
            prevScroll: 0, // предыдущее значение скролла
            adfoxHaveError: true, // Загрузилась ли реклама adfox
        };
    }

    componentDidMount() {
        window.addEventListener('scroll', this.onScroll);

        this.onScroll();
    }

    componentWillUnmount() {
        window.removeEventListener('scroll', this.onScroll);
    }

    onInputFocus = field => {
        clearTimeout(this._hideSuggestTimeout);
        this.showSuggest(field);
    };

    onChange = (field, value) => {
        const {dispatch} = this.props;

        reachGoal('search_form_change_parameters');

        if (field === 'from') {
            dispatch(setFromPointFromUser(value));
        } else if (field === 'to') {
            dispatch(setToPointFromUser(value));
        } else if (field === 'when') {
            dispatch(setWhen(value));
        }
    };

    /**
     * Смотри описание костыля в методе onOpenSearchForm
     */
    onScroll = () => {
        const currentScroll = window.pageYOffset;

        if (
            this.state.prevScroll !== currentScroll &&
            !this.context.bodyFixed
        ) {
            // изменяем состояние: position у формы поиска станет static
            this.setState({
                searchFormPostClose: false,
                searchFormTop: null,
                prevScroll: currentScroll,
            });
        }
    };

    /**
     * Изменяет состояние при "приливании" формы поиска к верху экрана
     * @param {boolean} headerIsSticky
     */
    onChangeHeaderIsSticky = headerIsSticky => {
        this.setState({
            headerIsSticky,
        });
    };

    /**
     * Описание костыля для красивой анимации:
     * Перед раскрытием формы поиска, для ее плавного перемещения вверх экрана, где ее стили
     * будут примерно седующими:
     *
     * position: fixed
     * top: 0
     *
     * нужно явно задать значение top для его анимации. Поэтому делаем промежуточное состояние,
     * которое задаст position: fixed и какой-то top, который мы запомним (state.searchFormTop)
     * и после сворачивания снова зададим.
     * Но как только произойдет прокрутка экрана мы вернем position у формы поиска в static.
     * @return {null}
     */
    onOpenSearchForm = () => {
        this.props.onOpen();

        if (this.state.headerIsSticky) {
            // если форма уже прилипла к верху экрана, то никаких костылей не требуется
            return null;
        }

        this.setState(
            {
                searchFormPreOpen: true,
                searchFormPostClose: false,
                searchFormTop: this._stickyContent.getBoundingClientRect().top,
            },
            () => {
                this.setState({
                    searchFormPreOpen: false,
                });
            },
        );
    };

    /**
     * Задает состояние последующее сворачиванию формы поиска.
     * Смотри описание костыля в методе onOpenSearchForm
     *
     * @param {boolean} isSubmit
     */
    onCloseSearchForm = (isSubmit = false) => {
        this.props.onClose();

        if (isSubmit) {
            // сбрасываем все костыльные состояния для анимации
            this.setState({
                searchFormPreOpen: false,
                searchFormPostClose: false,
                searchFormTop: null,
            });
        } else {
            // значение отступа у элемента Header_top
            const headerTopMarginBottom = parseFloat(
                window
                    .getComputedStyle(this._header, null)
                    .getPropertyValue('margin-bottom'),
            );

            this.setState({
                searchFormPostClose: true,
                searchFormTop:
                    this._header.getBoundingClientRect().bottom +
                    headerTopMarginBottom,
            });
        }
    };

    onClickParanja = () => this.onCloseSearchForm(false);

    onOpenDatepicker = () => {
        if (this.props.showOnlyOpenedForm) {
            this.context.fixBody();
        }
    };

    onCloseDatepicker = () => {
        if (this.props.showOnlyOpenedForm) {
            this.context.releaseBody();
        }
    };

    onSuggestSubmit = () => {
        this.hideSuggest();
    };

    // Если adfox не загрузился - нужно убрать margin (просто не рендерим блок с adFox)
    onAdFoxError = () => {
        this.setState({
            adfoxHaveError: false,
        });
    };

    static contextType = RootContext;

    /**
     * Возвращает высоту шапки, которая перекрывает содержимое страницы,
     * т.е. имеет position: fixed
     * чтобы можно было отскроллить окно браузера до какого-то видимого элемента на странице.
     * @return {number}
     */
    getHeightAt() {
        return !this.props.staticHeader && this.state.headerIsSticky
            ? HEIGHT_HEADER_STICKY
            : 0;
    }

    showSuggest = suggestField => {
        if (['from', 'to'].includes(suggestField)) {
            if (this.props.showOnlyOpenedForm) {
                this.context.fixBody();
            }

            // prevent ios autoscroll on input focus
            window.scrollTo(0, 1);
            this.setState({
                suggestField,
            });
        } else {
            this.hideSuggest();
        }
    };

    hideSuggest = () => {
        reachGoal('search_form_suggest_close_click');

        if (this.props.showOnlyOpenedForm) {
            this.context.releaseBody();
        }

        this.setState({
            suggestField: undefined,
        });
    };

    selectSuggest = value => {
        this.onChange(this.state.suggestField, value);
        this.hideSuggest();
    };

    renderParanja() {
        const {opened} = this.props;

        return (
            <div
                className={b('paranja', {opened})}
                onClick={this.onClickParanja}
            />
        );
    }

    renderLogoAndPersonal() {
        const {tld, language, user, page, clientSettlement} = this.props;

        return (
            <LogoAndPersonal
                tld={tld}
                language={language}
                user={user}
                page={page}
                clientSettlement={clientSettlement}
            />
        );
    }

    renderSearchForm() {
        const {suggestField} = this.state;
        const {opened, search, dispatch, showOnlyOpenedForm, pageType} =
            this.props;
        const {context} = search;

        const hideSelectorOfTransportType = [
            TRANSPORT_PAGE_NAME,
            TRANSPORT_CITY_PAGE_NAME,
        ].includes(pageType);

        return (
            <SearchForm
                className={b('searchForm')}
                inputsContainerRef={refCallback(this, '_inputsContainer')}
                opened={opened}
                minimized={opened}
                suggestField={suggestField}
                dispatch={dispatch}
                onOpen={this.onOpenSearchForm}
                onClose={this.onCloseSearchForm}
                onFocus={this.onInputFocus}
                onChange={this.onChange}
                onOpenDatepicker={this.onOpenDatepicker}
                onCloseDatepicker={this.onCloseDatepicker}
                onSuggestSubmit={this.onSuggestSubmit}
                context={context}
                showOnlyOpenedForm={showOnlyOpenedForm}
                hideSelectorOfTransportType={hideSelectorOfTransportType}
            />
        );
    }

    renderSuggest() {
        const {suggestField} = this.state;
        const {
            searchForm,
            suggests,
            language,
            nationalVersion,
            clientSettlement,
        } = this.props;
        const {transportType, userInput} = searchForm;

        if (!suggestField) {
            return null;
        }

        const dataProviderOptions = buildSearchFormSuggestDataProviderOptions({
            field: suggestField,
            suggests,
            language,
            nationalVersion,
            clientSettlement,
            from: userInput.from,
            to: userInput.to,
            transportType,
        });

        return (
            <GlobalPopup>
                <Suggest
                    language={language}
                    dataProviderOptions={dataProviderOptions}
                    inputData={searchForm}
                    suggestField={suggestField}
                    hide={this.hideSuggest}
                    select={this.selectSuggest}
                    onChange={this.onChange}
                    onSubmit={this.onSuggestSubmit}
                />
            </GlobalPopup>
        );
    }

    renderSearchFormWithSuggest() {
        const {
            staticHeader,
            currentSettlement,
            pageType,
            showOnlyOpenedForm,
            page: {fetching},
            transportCity: {
                transportType,
                settlement: {title: transportCityTitle},
            },
            transport,
            home,
            tld,
        } = this.props;
        const {isCity} = home;
        const {title, title_genitive: titleGenitive} = currentSettlement || {};

        if (showOnlyOpenedForm) {
            let headerText = null;

            switch (pageType) {
                case HOME_PAGE_NAME: {
                    if (isCity) {
                        const titleForKeyset = titleGenitive || title;

                        if (!titleForKeyset) {
                            headerText = sloganKeyset('text');
                            break;
                        }

                        headerText = sloganKeyset('text-city', {
                            titleGenitive: titleForKeyset,
                        });
                        break;
                    }

                    headerText = sloganKeyset('text');
                    break;
                }

                case TRANSPORT_CITY_PAGE_NAME: {
                    headerText = fetching
                        ? CHAR_NBSP
                        : transportCityKeyset(`title-${transportType}`, {
                              title: transportCityTitle,
                          });
                    break;
                }

                case TRANSPORT_PAGE_NAME: {
                    const {titlePrepositional} =
                        transport.countries.find(
                            country => country.code.toLowerCase() === tld,
                        ) || {};

                    headerText =
                        fetching || !titlePrepositional
                            ? CHAR_NBSP
                            : transportKeyset(
                                  `slogan-${transport.transportType}`,
                                  {titlePrepositional},
                              );
                    break;
                }
            }

            return (
                <div className={b('searchFormWithSuggest')}>
                    <h1 className={b('text')}>{headerText}</h1>
                    {this.renderSearchForm()}
                    {this.renderSuggest()}
                </div>
            );
        }

        const stickyContentStyle = {};

        if (this.state.searchFormPreOpen || this.state.searchFormPostClose) {
            // фиксируем форму для анимации (смотри метод onOpenSearchForm)
            stickyContentStyle.top = this.state.searchFormTop;
            stickyContentStyle.position = 'fixed';
        }

        return (
            <Sticky
                contentClassName={b('sticky')}
                static={staticHeader}
                onChange={this.onChangeHeaderIsSticky}
                contentRef={refCallback(this, '_stickyContent')}
                contentStyle={stickyContentStyle}
            >
                {this.renderSearchForm()}
                {this.renderSuggest()}
            </Sticky>
        );
    }

    renderTeasers() {
        const {teasers} = this.props;

        if (!teasers || teasers.length === 0) {
            return null;
        }

        return <Teasers className={b('teasers')} teasers={teasers} />;
    }

    renderBanner() {
        if (!this.props.renderBanner) {
            return null;
        }

        return (
            this.state.adfoxHaveError && (
                <AdFoxBanner
                    position={AdFoxPosition.center}
                    type={AdFoxType.inline}
                    className={b('adfoxBanner')}
                    onError={this.onAdFoxError}
                />
            )
        );
    }

    render() {
        const {opened, pageType, showOnlyOpenedForm} = this.props;

        return (
            <div className={b({opened, showOnlyOpenedForm, pageType})}>
                {!showOnlyOpenedForm && this.renderParanja()}

                {!showOnlyOpenedForm && (
                    <header
                        className={b('top')}
                        ref={refCallback(this, '_header')}
                    >
                        {this.renderLogoAndPersonal()}
                    </header>
                )}

                {this.renderSearchFormWithSuggest()}

                {this.renderTeasers()}

                {this.renderBanner()}

                <div className={b('forFixMarginBottom')}>0</div>
            </div>
        );
    }
}
