import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';

import {TRANSPORT_CITY_PAGE_NAME} from '../../../routes/transportCity';
import {TRANSPORT_PAGE_NAME} from '../../../routes/transport';
import {HOME_PAGE_NAME} from '../../../routes';

import Tld from '../../../interfaces/Tld';
import {TransportType} from '../../../lib/transportType';
import ICountry from '../../../interfaces/state/transport/ICountry';
import ISettlement from '../../../interfaces/state/settlement/ISettlement';

import keyset from '../../../i18n/slogan';
import transportCityKeyset from '../../../i18n/transport-city';
import transportKeyset from '../../../i18n/transport';

const b = B('Slogan');

interface ISloganProps {
    pageType: string;
    tld: Tld;

    transportType?: TransportType;
    transportCityType?: TransportType;
    countries?: ICountry[];
    currentSettlement?: ISettlement;
    hidden?: boolean;
    isCity?: boolean;
    showAppearAnimation?: boolean;
    noAnimation?: boolean;
}

export default class Slogan extends PureComponent<ISloganProps, {}> {
    sloganText = '';

    static defaultProps = {
        transportType: '',
        transportCityType: '',
        currentSettlement: {},
        hidden: false,
        showAppearAnimation: false,
    };

    getSloganText(): string {
        const {
            pageType,
            transportType,
            transportCityType,
            countries,
            tld,
            isCity,
            currentSettlement,
        } = this.props;
        const {title, title_genitive: titleGenitive} = currentSettlement || {};

        switch (pageType) {
            case TRANSPORT_CITY_PAGE_NAME:
                if (!title) {
                    return '';
                }

                return transportCityKeyset(`title-${transportCityType}`, {
                    title,
                });
            case TRANSPORT_PAGE_NAME: {
                if (!countries) {
                    return '';
                }

                const countryToDisplay = countries.find(
                    country => country.code.toLowerCase() === tld,
                );

                if (!countryToDisplay) {
                    return '';
                }

                const {titlePrepositional, title: countryTitle} =
                    countryToDisplay;

                return transportKeyset(`slogan-${transportType}`, {
                    titlePrepositional: titlePrepositional || countryTitle,
                });
            }

            case HOME_PAGE_NAME:
                if (isCity) {
                    const titleForKeyset = titleGenitive || title;

                    if (!titleForKeyset) {
                        return keyset('text');
                    }

                    return keyset('text-city', {titleGenitive: titleForKeyset});
                }

                return keyset('text');

            default:
                return keyset('text');
        }
    }

    render(): React.ReactElement {
        const {hidden, showAppearAnimation, noAnimation} = this.props;

        // Если элемента нет на странице, берем закэшированное значение
        const text = hidden ? this.sloganText : this.getSloganText();

        this.sloganText = text;

        return (
            <div
                className={b({
                    hidden,
                    appear: showAppearAnimation,
                    noAnimation,
                })}
            >
                {/* Для SEO мы убираем h1 со страницы если текст скрыт
                    Для плавности анимаций оставляем span */}
                {hidden ? <span>{text}</span> : <h1>{text}</h1>}
            </div>
        );
    }
}
