import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';

import Tld from '../../../interfaces/Tld';
import Lang from '../../../interfaces/Lang';
import IconGlyph from '../../../interfaces/components/IconGlyph';
import IStation from '../../../interfaces/state/stationsGroup/IStation';
import {IStationWithAeroexpress} from '../../../interfaces/state/stationsGroup/IStationWithAeroexpress';

import {
    filterAirports,
    trueAeroexpress,
    getUrlForAirport,
    getAeroexpressListUrl,
} from '../../../lib/aeroexpress';

import Link from '../../Link';
import Icon from '../../Icon/Icon';
import Dropdown from '../../Dropdown/Dropdown';

import keyset from '../../../i18n/bt-lib-aeroexpress';
import calendarKeyset from '../../../i18n/calendar';

const b = B('Aeroexpress');
const isMobile = process.env.PLATFORM === 'mobile';

interface IAeroexpressProps {
    stations: IStation[];
    tld: Tld;
    language: Lang;

    cityId?: number;
}

class Aeroexpress extends PureComponent<IAeroexpressProps> {
    getSingleTitle(hasTrueAeroexpress): string {
        return keyset(
            hasTrueAeroexpress ? 'single-title' : 'single-suburban-title',
        );
    }

    getSingleFullTitle(hasTrueAeroexpress, airport): string {
        return keyset(
            hasTrueAeroexpress
                ? 'single-full-title'
                : 'single-full-suburban-title',
            {airportTitle: airport.originalTitle},
        );
    }

    getContent(
        hasAirportsWithoutAeroexpress: boolean,
        airports: IStationWithAeroexpress[],
        cityId?: number,
    ): React.ReactNode {
        const {tld, language} = this.props;
        const hasTrueAeroexpress = trueAeroexpress(airports);

        if (isMobile && cityId) {
            return (
                <div className={b('content')}>
                    {hasTrueAeroexpress && (
                        <Icon
                            className={b('icon')}
                            glyph={IconGlyph.aeroexpressLogo}
                        />
                    )}

                    <Link
                        className={b('link')}
                        href={getAeroexpressListUrl(cityId, tld, language)}
                    >
                        {this.getSingleTitle(hasTrueAeroexpress)}
                    </Link>
                </div>
            );
        }

        const when = calendarKeyset('all-days');

        if (airports.length === 1) {
            return (
                <div className={b('content')}>
                    <Link
                        className={b('link')}
                        href={getUrlForAirport(
                            airports[0],
                            hasTrueAeroexpress,
                            when,
                            tld,
                            language,
                        )}
                    >
                        {hasAirportsWithoutAeroexpress
                            ? this.getSingleFullTitle(
                                  hasTrueAeroexpress,
                                  airports[0],
                              )
                            : this.getSingleTitle(hasTrueAeroexpress)}
                    </Link>
                </div>
            );
        }

        const buttonText = keyset(
            hasTrueAeroexpress ? 'plural-title' : 'plural-suburban-title',
        );

        return (
            <div className={b('content')}>
                <Dropdown
                    className={b('dropdown')}
                    noArrow
                    buttonText={buttonText}
                    size="small"
                >
                    <ul className={b('list')}>
                        {airports
                            .filter(airport => airport.aeroexpress)
                            .map(airport => (
                                <li className={b('airport')} key={airport.id}>
                                    <Link
                                        className={b('airport-link')}
                                        href={getUrlForAirport(
                                            airport,
                                            hasTrueAeroexpress,
                                            when,
                                            tld,
                                            language,
                                        )}
                                    >
                                        {airport.title}
                                    </Link>
                                </li>
                            ))}
                    </ul>
                </Dropdown>
            </div>
        );
    }

    render(): React.ReactNode {
        const {stations, cityId} = this.props;

        const stationsWithAeroexpress = filterAirports(stations);
        const hasAirportsWithoutAeroexpress =
            stationsWithAeroexpress.length < stations.length;

        return (
            <div className={b({isMobile})}>
                {this.getContent(
                    hasAirportsWithoutAeroexpress,
                    stationsWithAeroexpress,
                    cityId,
                )}
            </div>
        );
    }
}

export default Aeroexpress;
