import {refCallback} from '../../base';

import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';
import changeUrl from 'page';
import {connect} from 'react-redux';

import {CITY} from '../../../lib/constants/suggestOptionTypes';

import KeyCode from '../../../interfaces/KeyCode';
import IconGlyph from '../../../interfaces/components/IconGlyph';

import defaultDataProvider from '../../../lib/suggests/defaultDataProvider';
import {reachGoalOnce} from '../../../lib/yaMetrika';
import tuneRegionUrl from '../../../lib/url/tuneRegionUrl';
import getDataProviderOptions from '../../../lib/suggests/getDataProviderOptions';
import cityUrl from '../../../lib/url/getCityUrl';

import Icon from '../../Icon/Icon';
import Button from '../../Button/Button';
import Button2 from '../../Button2/Button2';
import Popup from '../../Popup/Popup';
import PopupError from '../../PopupError/PopupError';
import Input from '../../Input/Input';
import Link from '../../Link';
import PointSuggestDrop from '../../PointSuggestDrop/PointSuggestDrop';

import formKeyset from '../../../i18n/city-change-form';
import tuneRegionKeyset from '../../../i18n/tune-region';

const b = B('ChangeCity');

const mapStateToProps = ({language}) => ({language});

class ChangeCity extends PureComponent {
    static defaultProps = {
        value: {
            title: '',
            key: '',
        },
        dataProvider: defaultDataProvider,
        dataProviderOptions: {
            url: '//suggests.rasp.yandex.net',
            path: 'all_suggests',
            query: {
                format: 'old',
                lang: 'ru',
                client_city: 213,
                national_version: 'ru',
            },
        },
    };

    state = {
        popupVisible: false,
        value: '',
        items: [],
        selectedItemIndex: -1,
        idToApply: -1,
        error: false,
    };

    onClick = () => {
        this.setState({popupVisible: true});
    };

    onClose = () => {
        this.setState({
            popupVisible: false,
            selectedItemIndex: -1,
            error: false,
        });
        this.input.blur();
    };

    onSubmit = e => {
        const {value, idToApply} = this.state;

        if (!value) {
            e.preventDefault();

            this.setState({error: true});

            return;
        }

        if (idToApply > -1) {
            e.preventDefault();

            this.setState({value: ''});
            this.onClose();

            changeUrl(cityUrl(idToApply));
        }
    };

    onShowPopup = () => {
        setTimeout(() => {
            this.input.focus();
        }, 100);
    };

    onInputChange = newValue => {
        this.setState({
            selectedItemIndex: -1,
            idToApply: -1,
            value: newValue,
        });
        this.requestItems(newValue);
    };

    onInputFocus = () => {
        const {value} = this.state;

        this.requestItems(value);
        this.setState({error: false});
    };

    onDropItemMouseOver = (e, data) => {
        this.setState({
            selectedItemIndex: this.getItemIndexByKey(data.value.key),
        });
    };

    onDropItemClick = (e, data) => {
        const {key, title} = data.value;

        this.setState({
            idToApply: key.substr(1),
            value: title,
            error: false,
        });
    };

    onRegionTuneClick() {
        reachGoalOnce('tune_region_click');
    }

    onKeyDown = e => {
        switch (e.keyCode) {
            case KeyCode.up:
                this.shiftItemSelection(e, -1);
                break;
            case KeyCode.down:
                this.shiftItemSelection(e, 1);
                break;
            case KeyCode.esc:
                this.setState({popupVisible: false});
        }
    };

    getTitleButton = currentSettlement => {
        const arrow = (
            <Icon className={b('arrow')} glyph={IconGlyph.geoArrow} />
        );

        return (
            <Button
                onClick={this.onClick}
                leftIcon={arrow}
                className={b('title')}
            >
                {currentSettlement.title}
            </Button>
        );
    };

    getItemIndexByKey(key) {
        const items = this.state.items;

        for (let i = 0; i < items.length; i++) {
            if (key === items[i].value.key) {
                return i;
            }
        }

        return -1;
    }

    shiftItemSelection = (e, shift) => {
        const {items} = this.state;
        const length = items.length;

        e.preventDefault();

        if (length > 0) {
            let selectedItemIndex = this.state.selectedItemIndex;

            if (selectedItemIndex === -1 && shift < 0) {
                selectedItemIndex = length;
            }

            selectedItemIndex += shift;

            if (selectedItemIndex >= length || selectedItemIndex < 0) {
                this.setState({
                    selectedItemIndex: -1,
                    idToApply: -1,
                    value: '',
                });

                return;
            }

            const {key, title} = items[selectedItemIndex].value;

            this.setState({
                selectedItemIndex,
                idToApply: key.substr(1),
                value: title,
                error: false,
            });
        }
    };

    requestItems(text) {
        const {
            dataProvider,
            suggests,
            nationalVersion,
            language,
            clientSettlement,
        } = this.props;

        const dataProviderOptions = getDataProviderOptions({
            suggests,
            nationalVersion,
            settlement: clientSettlement,
            language,
            path: CITY,
        });

        dataProvider(text, dataProviderOptions).then(items => {
            if (this.state.value === text) {
                this.setState({items});
            }
        });
    }

    render() {
        const {currentSettlement, clientSettlement, tld, page} = this.props;
        const {popupVisible, error, value} = this.state;

        return (
            <div className={b()}>
                {this.getTitleButton(currentSettlement)}

                <Popup
                    visible={popupVisible}
                    onShow={this.onShowPopup}
                    onClickOutside={this.onClose}
                    className={b('suggestPopup')}
                >
                    <div className={b('popupContent')}>
                        <Button
                            className={b('closeButton')}
                            onClick={this.onClose}
                        >
                            <Icon
                                className={b('closeIcon')}
                                glyph={IconGlyph.cross}
                            />
                        </Button>

                        <form
                            className={b('form')}
                            action="/station_search/"
                            onSubmit={this.onSubmit}
                        >
                            <div className={b('input')}>
                                <Popup
                                    className={b('errorPopup')}
                                    visible={error && popupVisible}
                                >
                                    <PopupError>
                                        {formKeyset('error-city-empty')}
                                    </PopupError>
                                </Popup>

                                <Input
                                    onChange={this.onInputChange}
                                    onKeyDown={this.onKeyDown}
                                    onFocus={this.onInputFocus}
                                    name="cityFrom"
                                    value={value}
                                    inputRefCallback={refCallback(
                                        this,
                                        'input',
                                    )}
                                    autoComplete={false}
                                    autoCorrect={false}
                                />
                            </div>

                            <Button2
                                className={b('changeButton')}
                                type="submit"
                            >
                                {formKeyset('submit')}
                            </Button2>
                        </form>

                        <PointSuggestDrop
                            className={b('suggest')}
                            items={this.state.items}
                            visible={popupVisible}
                            selectedItemIndex={this.state.selectedItemIndex}
                            onItemMouseOver={this.onDropItemMouseOver}
                            onItemClick={this.onDropItemClick}
                            paddingSides={20}
                        />

                        <div className={b('currentCity')}>
                            {`${tuneRegionKeyset('text')} ${
                                clientSettlement.title
                            } `}
                            <Link
                                className={b('tuneRegion')}
                                onClick={this.onRegionTuneClick}
                                href={tuneRegionUrl(tld, {
                                    retpath: page.fullUrl,
                                })}
                                title={clientSettlement.title}
                            >
                                {tuneRegionKeyset('link')}
                            </Link>
                        </div>
                    </div>
                </Popup>
            </div>
        );
    }
}

export default connect(mapStateToProps)(ChangeCity);
