import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';
import page from 'page';

import {STATION} from '../../../lib/constants/suggestOptionTypes';

import IconGlyph from '../../../interfaces/components/IconGlyph';
import Lang from '../../../interfaces/Lang';
import TPointSuggestItem from '../../../interfaces/components/PointSuggest/TPointSuggestItem';
import TPointSuggestError from '../../../interfaces/components/PointSuggest/TPointSuggestError';
import TPointSuggestCallback from '../../../interfaces/components/PointSuggest/TPointSuggestCallback';
import NationalVersion from '../../../interfaces/NationalVersion';
import ISettlement from '../../../interfaces/state/settlement/ISettlement';
import IStateSuggests from '../../../interfaces/state/IStateSuggests';
import Tld from '../../../interfaces/Tld';
import PointType from '../../../interfaces/PointType';

import getDataProviderOptions from '../../../lib/suggests/getDataProviderOptions';
import getPointId from '../../../lib/point/getPointId';
import getPointType from '../../../lib/point/getPointType';
import {stationUrl} from '../../../lib/url/stationUrl';

import PointSuggest from '../../PointSuggest/PointSuggest';
import Icon from '../../Icon/Icon';
import Button from '../../Button/Button';

import keyset from '../../../i18n/station-route-search-form';

const b = B('StationSearch');

interface IStationSearchProps {
    language: Lang;
    nationalVersion: NationalVersion;
    settlement: ISettlement;
    suggests: IStateSuggests;
    tld: Tld;
}

interface IStationSearchState {
    value: TPointSuggestItem;
    errors: TPointSuggestError[];
}

export default class StationSearch extends PureComponent<
    IStationSearchProps,
    IStationSearchState
> {
    state: IStationSearchState = {
        value: {
            title: '',
        },
        errors: [],
    };

    formNode = React.createRef<HTMLFormElement>();

    onChange: TPointSuggestCallback = (e, {value}) => {
        this.setState({
            value,
        });

        if (this.state.errors.length) {
            this.setErrors([]);
        }
    };

    onSubmit = (e: React.FormEvent<HTMLFormElement>): void => {
        e.preventDefault();

        this.onApply();
    };

    onFocus = (): void => {
        this.setErrors([]);
    };

    onApply = (): void => {
        const {title, key} = this.state.value;

        // Обрабатываем случай, когда известен id станции
        if (key && getPointType(key) === PointType.station) {
            const {tld, language} = this.props;
            const stationId = getPointId(key);

            return page.show(
                stationUrl({
                    id: stationId,
                    tld,
                    language,
                }),
            );
        }

        if (title) {
            // eslint-disable-next-line no-unused-expressions
            this.formNode.current?.submit();
        } else {
            this.setErrors([
                {key: 'empty-input', title: keyset('error-number-empty')},
            ]);
        }
    };

    setErrors(errors: TPointSuggestError[]): void {
        this.setState({
            errors,
        });
    }

    render(): React.ReactElement {
        const {value, errors} = this.state;

        const {nationalVersion, settlement, language, suggests} = this.props;

        const dataProviderOptionsObject = {
            suggests,
            nationalVersion,
            settlement,
            language,
            path: STATION,
        };

        return (
            <form
                onSubmit={this.onSubmit}
                method="GET"
                action="/station-route-search/"
                className={b()}
                ref={this.formNode}
            >
                <PointSuggest
                    id="suggest-stations"
                    name="query"
                    value={value}
                    errors={errors}
                    onChange={this.onChange}
                    onApply={this.onApply}
                    onFocus={this.onFocus}
                    placeholder={keyset('placeholder')}
                    dataProviderOptions={getDataProviderOptions(
                        dataProviderOptionsObject,
                    )}
                />

                <Button className={b('search-icon-wrapper')} type="submit">
                    <Icon
                        className={b('search-icon')}
                        glyph={IconGlyph.search}
                    />
                </Button>
            </form>
        );
    }
}
