import {React, PureComponent, B, refCallback} from '../../base';
import page from 'page';

import {STATION} from '../../../lib/constants/suggestOptionTypes';

import IconGlyph from '../../../interfaces/components/IconGlyph';
import PointType from '../../../interfaces/PointType';

import getDataProviderOptions from '../../../lib/suggests/getDataProviderOptions';
import getPointType from '../../../lib/point/getPointType';
import getPointId from '../../../lib/point/getPointId';
import {stationUrl as getStationUrl} from '../../../lib/url/stationUrl';

import RootContext from '../../Root/RootContext';
import Icon from '../../Icon/Icon';
import Button from '../../Button/Button';
import Suggest from '../../Suggest/Suggest';
import GlobalPopup from '../../GlobalPopup';

import keyset from '../../../i18n/station-route-search-form';

const b = B('StationSearch');

export default class StationSearch extends PureComponent {
    state = {
        suggestField: null,
        userInput: {
            title: '',
            key: '',
        },
    };

    onChange = (field, value) => {
        this.setState({
            userInput: value,
        });
    };

    onClick = () => {
        this.showSuggest();
    };

    onSuggestSubmit = () => {
        this.formNode.submit();
    };

    static contextType = RootContext;

    selectSuggest = value => {
        const {title, key} = value;

        // Обрабатываем случай, когда известен id станции
        if (key && getPointType(key) === PointType.station) {
            const {tld, language} = this.props;

            const stationUrl = getStationUrl({
                id: getPointId(key),
                tld,
                language,
            });

            return this.hideSuggest(() => {
                /* Из-за того, что манипуляции с историей браузера асинхронные (они используются
                    в Suggest через ModalHistoryWrapper для закрытия саджестов кнопкой
                    "назад" браузера), приходится сначала дожидаться изменения стэйта данного
                    компонента, который запустит манипуляции с историей браузера и уже потом
                    через n-секунд менять урл страницы на новый */
                setTimeout(() => page.show(stationUrl), 10);
            });
        }

        this.setState({userInput: {title, key}}, () => {
            if (title) {
                this.formNode.submit();
            }
        });

        this.hideSuggest();
    };

    showSuggest = () => {
        this.context.fixBody();

        // prevent ios autoscoll on input focus
        window.scrollTo(0, 1);
        this.setState({
            suggestField: STATION,
        });
    };

    hideSuggest = callback => {
        const preparedCallback =
            typeof callback === 'function' ? callback : undefined;

        this.context.releaseBody();
        this.setState(
            {
                suggestField: undefined,
            },
            preparedCallback,
        );
    };

    getInputData = () => {
        const {userInput} = this.state;

        return {
            userInput: {
                station: userInput,
            },
        };
    };

    render() {
        const {userInput, suggestField} = this.state;

        const {key: stationId, title} = userInput;

        const {nationalVersion, settlement, suggests, language} = this.props;

        const dataProviderOptionsObject = {
            suggests,
            nationalVersion,
            settlement,
            language,
            path: STATION,
        };

        return (
            <div className={b()}>
                <form
                    method="GET"
                    action="/station-search/"
                    className={b('form')}
                    ref={refCallback(this, 'formNode')}
                >
                    <input value={stationId} name="pointId" type="hidden" />

                    <input value={title} name="pointName" type="hidden" />

                    <Button
                        onClick={this.onClick}
                        className={b('search-button')}
                        rightIcon={
                            <Icon
                                className={b('search-icon')}
                                glyph={IconGlyph.search}
                            />
                        }
                    >
                        {keyset('placeholder-touch')}
                    </Button>
                </form>
                {suggestField && (
                    <GlobalPopup>
                        <Suggest
                            dataProviderOptions={getDataProviderOptions(
                                dataProviderOptionsObject,
                            )}
                            inputData={this.getInputData()}
                            suggestField={suggestField}
                            hide={this.hideSuggest}
                            select={this.selectSuggest}
                            formKeyset={keyset}
                            onChange={this.onChange}
                            inputName="pointName"
                            onSubmit={this.onSuggestSubmit}
                        />
                    </GlobalPopup>
                )}
            </div>
        );
    }
}
