import React, {
    memo,
    ReactElement,
    MutableRefObject,
    useState,
    useCallback,
    ChangeEvent,
    KeyboardEvent,
    FocusEvent,
    useEffect,
} from 'react';
import B from 'bem-cn-lite';

import Platform from '../../interfaces/Platform';
import IconGlyph from '../../interfaces/components/IconGlyph';

import getThrottle from '../../lib/getThrottle';

import Icon from '../Icon/Icon';

const b = B('Input');

const DEFAULT_SIZE_MOD = process.env.PLATFORM === Platform.mobile ? 'l' : 'm';

interface IInput {
    value: string;
    onChange: (newValue: string) => void;

    className?: string;
    throttleOnChange?: number; // Троттлинг события onChange
    inputRefCallback?: (
        element: HTMLInputElement | HTMLTextAreaElement | null,
    ) => void; // Вызовется с dom-элементом инпута, чтобы можно было им манипулирувать из внешнего кода, например вызвать focus()
    inputRef?: MutableRefObject<HTMLInputElement | null>;
    textareaRef?: MutableRefObject<HTMLTextAreaElement | null>;
    onKeyDown?: (
        event: KeyboardEvent<HTMLInputElement | HTMLTextAreaElement>,
    ) => void;
    onFocus?: (
        event: FocusEvent<HTMLInputElement | HTMLTextAreaElement>,
    ) => void;
    onBlur?: (
        event: FocusEvent<HTMLInputElement | HTMLTextAreaElement>,
    ) => void;
    onReset?: () => void; // Вызовится при клике на крести в правой части поля
    name?: string;
    autoComplete?: boolean;
    autoCorrect?: boolean;
    disabled?: boolean;
    size?: number;
    placeholder?: string;
    withSearchIcon?: boolean;
    sizeMod?: 'm' | 'l';
    widthMod?: '100'; // 100 - width:100%
    withReset?: boolean; // Добавляет механику сброса введенного значения по клику на крестик в правой части поля

    // Свойства для textarea
    textarea?: boolean;
    cols?: number;
    rows?: number;
}

export default memo(Input);

function Input({
    value: valueFromProps,
    onChange: onChangeFromProps,

    className,
    throttleOnChange = 300,
    inputRefCallback,
    inputRef,
    textareaRef,
    onKeyDown,
    onFocus,
    onBlur: onBlurFromProps,
    onReset,
    name,
    autoComplete = true,
    autoCorrect = true,
    disabled,
    size,
    placeholder,
    withSearchIcon = false,
    sizeMod = DEFAULT_SIZE_MOD,
    widthMod,
    withReset = false,
    textarea,
    cols,
    rows,
}: IInput): ReactElement {
    const [value, setValue] = useState(valueFromProps);

    const onChangeFromPropsThrottled = useCallback(
        getThrottle((newValue: string) => {
            onChangeFromProps(newValue);
        }, throttleOnChange),
        [onChangeFromProps],
    );

    const onChange = useCallback(
        (e: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
            const newValue = e.currentTarget.value;

            setValue(newValue);
            onChangeFromPropsThrottled(newValue);
        },
        [onChangeFromPropsThrottled],
    );

    const onBlur = useCallback(
        (event: FocusEvent<HTMLInputElement | HTMLTextAreaElement>) => {
            onChangeFromPropsThrottled.flush();

            if (onBlurFromProps) {
                onBlurFromProps(event);
            }
        },
        [onBlurFromProps, onChangeFromPropsThrottled],
    );

    const onClickReset = useCallback(() => {
        setValue('');
        onChangeFromProps('');

        if (onReset) {
            onReset();
        }
    }, [onChangeFromProps, onReset]);

    useEffect(() => {
        setValue(valueFromProps);
    }, [valueFromProps]);

    useEffect(
        () => () => {
            onChangeFromPropsThrottled.flush();
        },
        [onChangeFromPropsThrottled],
    );

    const icon = getIcon(value, withReset, withSearchIcon, onClickReset);

    return (
        <span
            className={b(
                {
                    size: sizeMod,
                    width: widthMod,
                    withIcon: Boolean(icon),
                },
                className,
            )}
        >
            {textarea ? (
                <textarea
                    className={b('input')}
                    onChange={onChange}
                    onBlur={onBlur}
                    value={value}
                    ref={textareaRef || inputRefCallback}
                    onKeyDown={onKeyDown}
                    onFocus={onFocus}
                    name={name}
                    autoComplete={!autoComplete ? 'off' : undefined}
                    autoCorrect={!autoCorrect ? 'off' : undefined}
                    disabled={disabled}
                    cols={cols}
                    rows={rows}
                    placeholder={placeholder}
                />
            ) : (
                <input
                    className={b('input')}
                    onChange={onChange}
                    onBlur={onBlur}
                    value={value}
                    ref={inputRef || inputRefCallback}
                    onKeyDown={onKeyDown}
                    onFocus={onFocus}
                    name={name}
                    autoComplete={!autoComplete ? 'off' : undefined}
                    autoCorrect={!autoCorrect ? 'off' : undefined}
                    disabled={disabled}
                    size={size}
                    placeholder={placeholder}
                />
            )}

            {icon}
        </span>
    );
}

function getIcon(
    value: string,
    withReset: boolean,
    withSearchIcon: boolean,
    onClickReset: () => void,
): ReactElement | null {
    if (withReset && value.length) {
        return (
            <span className={b('resetContainer')} onClick={onClickReset}>
                <Icon glyph={IconGlyph.crossBold} className={b('resetIcon')} />
            </span>
        );
    }

    if (withSearchIcon) {
        return <Icon glyph={IconGlyph.search} className={b('searchIcon')} />;
    }

    return null;
}
