import {PureComponent} from '../base';

interface IModalHistoryWrapperProps {
    onClose: () => void;
    children: React.ReactElement;
}

/*
    Компонент-обертка.
    Использование:
        В него нужно обернуть содержимое, которое рендерится, только когда явно показывается пользователю,
        и удаляется из dom, если этот элемент скрыли или кликнули по кнопке "назад" браузера.
*/
export default class ModalHistoryWrapper extends PureComponent<
    IModalHistoryWrapperProps,
    {}
> {
    popstateWasCalled = false;
    lastPath = location.pathname + location.search;
    lastUrl = window.location.href;

    componentDidMount(): void {
        if (typeof window === 'undefined') {
            return;
        }

        const {path, url} = this.getPathAndUrl();

        // Добавление fake = true не будет приводить к перезагрузке страницы (см. common/middleware/preventRefreshPage)
        this.replaceState(true);
        history.pushState({fake: true, path}, '', url);

        window.addEventListener('popstate', this.popstateHandler);
    }

    componentWillUnmount(): void {
        window.removeEventListener('popstate', this.popstateHandler);

        if (this.popstateWasCalled === false) {
            // До закрытия попапа popstateHandler не вызывался, значит попап
            // был закрыт не аппаратной кнопкой назад, а каким-то иным образом, например кнопкой внутри попапа
            // Нужно обработать этот случай так, чтобы следующее нажатие на аппаратную кнопку назад вело нас
            // не на копию history, а на предыдущую страницу.
            window.addEventListener('popstate', this.popstateHandler2);
            this.memoizePath();
            history.back(); // Этот метод асинхронный, из-за чего приходится изголяться с регистацией слушателя popstateHandler2
        }
    }

    // Событие popstate вызывается каждый раз, когда история изменятся.
    // если запись истории была создана с помощью вызова pushState или replaceState
    popstateHandler = (): void => {
        this.replaceState();

        this.popstateWasCalled = true;
        this.props.onClose();
    };

    popstateHandler2 = (): void => {
        this.replaceState();

        window.removeEventListener('popstate', this.popstateHandler2);
    };

    memoizePath = (): void => {
        this.lastPath = location.pathname + location.search;
        this.lastUrl = window.location.href;
    };

    getPathAndUrl = (): {path: string; url: string} => ({
        path: this.lastPath,
        url: this.lastUrl,
    });

    replaceState = (fake = false): void => {
        const {path, url} = this.getPathAndUrl();

        history.replaceState({fake, path}, '', url);
    };

    render(): React.ReactElement {
        return this.props.children;
    }
}
