import React, {PureComponent} from 'react';
import {connect} from 'react-redux';

import {isDateInRange} from '../../lib/date/utils';
import getTrainOrderUrl from '../../lib/url/getTrainOrderUrl';
import getUfsBuyLink from '../../lib/url/ufs-buy';
import {dayMatchesMask} from '../../lib/date/mask';
import {addOrderGoalHandler} from '../../lib/metrika/createOrderGoalHandler';
import shouldUseLinkToTrains from '../../lib/order/shouldUseLinkToTrains';
import isTransferSegment from '../../lib/segments/isTransferSegment';
import noop from '../../lib/noop';

import SegmentBuyButtonUtmContext from '../../contexts/SegmentBuyButtonUtmContext';
import CalendarDayLink from '../CalendarDayLink';
import ThreadCalendarDayLink from '../ThreadCalendarDayLink/ThreadCalendarDayLink';

const mapStateToProps = ({tld, environment, flags, seoQueryParams, user}) => ({
    tld,
    environment,
    flags,
    seoQueryParams,
    clientId: user.clientId,
});

class OrderCalendarDayLink extends PureComponent {
    static contextType = SegmentBuyButtonUtmContext;

    render() {
        const {
            environment: {production: isProduction},
            tld,
            flags,
            seoQueryParams,
            clientId,
            day,
            today,
            orderRange,
            onClick,
            context,
            segment,
            tariffClassKeys,
            language,
            currencies,
            nowMoment,
        } = this.props;
        const {getUtmMedium} = this.context;

        const {
            doNotSell,
            transport: {code: transportType},
        } = segment;

        if (doNotSell || !isDateInRange(day, orderRange)) {
            return (
                <ThreadCalendarDayLink
                    day={day}
                    today={today}
                    onClick={onClick}
                    context={context}
                    segment={segment}
                    language={language}
                />
            );
        }

        let href;
        let clickHandler = noop();

        if (dayMatchesMask(day, segment.runDays)) {
            /**
             * Изначально у сегмента время отправления минимальное из возможных.
             * Например, на выдаче на все дни есть три дня, когда едет поезд: 2017-08-01, 2017-08-02 и 2017-08-03.
             * departureMoment будет соответствовать 2017-08-01.
             * Поэтому изменяем время, чтобы правильно сформировать ссылку.
             */
            const departure = segment.departureMoment
                .clone()
                .year(day.year())
                .month(day.month())
                .date(day.date());
            const useLinkToTrains = shouldUseLinkToTrains(segment);

            if (departure.isBefore(nowMoment)) {
                return (
                    <ThreadCalendarDayLink
                        day={day}
                        today={today}
                        onClick={onClick}
                        context={context}
                        segment={segment}
                        language={language}
                    />
                );
            }

            // TODO: убрать такое формирование ссылки, когда продажа будет только на стороне Поездов,
            // нужно будет парсить ссылку, которая будет приходить с бэка и заменять в ней дату
            href = useLinkToTrains
                ? getTrainOrderUrl({
                      coachType: tariffClassKeys[0],
                      tld,
                      flags,
                      context,
                      segment,
                      currencies,
                      departure,
                      isProduction,
                      seoQueryParams,
                      clientId,
                      utmMedium: getUtmMedium(
                          transportType,
                          isTransferSegment(segment),
                      ),
                  })
                : getUfsBuyLink(segment, day);

            clickHandler = addOrderGoalHandler(
                onClick,
                segment.transport.code,
                null,
                !useLinkToTrains,
                segment.badges,
            );
        }

        return (
            <CalendarDayLink
                day={day}
                href={href}
                today={today}
                rel="nofollow"
                target="_blank"
                onClick={clickHandler}
            />
        );
    }
}

export default connect(mapStateToProps)(OrderCalendarDayLink);
