import React, {ReactElement, memo, useContext} from 'react';
import {Moment} from 'moment';

import {TransportType} from '../../lib/transportType';
import ISearchContext from '../../interfaces/state/search/ISearchContext';
import ISegment from '../../interfaces/segment/ISegment';
import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
import CalendarOrderRange from '../../interfaces/components/calendar/CalendarOrderRange';

import planeOrderUrl from '../../lib/url/planeOrderUrl';
import {isDateInRange} from '../../lib/date/utils';
import {dayMatchesMask} from '../../lib/date/mask';
import isTransferSegment from '../../lib/segments/isTransferSegment';
import {addOrderGoalHandler} from '../../lib/metrika/createOrderGoalHandler';
import useSelector from '../useSelector';

import SegmentBuyButtonUtmContext from '../../contexts/SegmentBuyButtonUtmContext';
import CalendarDayLink from '../CalendarDayLink';
import ThreadCalendarDayLink from '../ThreadCalendarDayLink/ThreadCalendarDayLink';

interface IPlaneCalendarDayLink {
    day: Moment;
    today: Moment;
    orderRange: CalendarOrderRange;
    onClick: () => void;
    context: ISearchContext;
    segment: ISegment;
}

export default memo(PlaneCalendarDayLink);

function PlaneCalendarDayLink({
    day,
    today,
    orderRange,
    onClick,
    context,
    segment,
}: IPlaneCalendarDayLink): ReactElement {
    const {getUtmMedium} = useContext(SegmentBuyButtonUtmContext);
    const seoQueryParams = useSelector(state => state.seoQueryParams);
    const language = useSelector(state => state.language);
    const clientId = useSelector(state => state.user.clientId);
    const isProduction = useSelector(state => state.environment.production);

    const {
        number,
        departureMoment,
        transport: {code: transportType},
    } = segment;

    let href: string | undefined;

    if (!isDateInRange(day, orderRange)) {
        return (
            <ThreadCalendarDayLink
                day={day}
                today={today}
                onClick={onClick}
                context={context}
                segment={segment}
            />
        );
    }

    if (departureMoment && dayMatchesMask(day, segment.runDays)) {
        const isAllDaysSearch =
            context.when.special === DateSpecialValue.allDays;

        href = planeOrderUrl({
            to: context.to,
            from: context.from,
            when: day,
            departure: departureMoment,
            number,
            seoQueryParams,
            utmMedium: getUtmMedium(
                transportType,
                isTransferSegment(segment),
                isAllDaysSearch,
            ),
            clientId,
            language,
            isProduction,
        });
    }

    return (
        <CalendarDayLink
            day={day}
            href={href}
            today={today}
            rel="nofollow"
            target="_blank"
            onClick={addOrderGoalHandler(
                onClick,
                TransportType.plane,
                undefined,
                undefined,
                segment.badges,
            )}
        />
    );
}
