import React from 'react';
import B from 'bem-cn-lite';

import KeyCode from '../../interfaces/KeyCode';
import SecondaryPosition from '../../interfaces/lib/dimensions/SecondaryPosition';
import PrimaryPosition from '../../interfaces/lib/dimensions/PrimaryPosition';
import TPointSuggestItem from '../../interfaces/components/PointSuggest/TPointSuggestItem';
import TPointSuggestCallback from '../../interfaces/components/PointSuggest/TPointSuggestCallback';
import Point from '../../interfaces/Point';
import ISuggestDataProviderOptions from '../../interfaces/lib/suggests/ISuggestDataProviderOptions';
import TSuggestDataProvider from '../../interfaces/lib/suggests/TSuggestDataProvider';
import TSuggestItemFromBackend from '../../interfaces/lib/suggests/TSuggestItemFromBackend';
import TPointSuggestError from '../../interfaces/components/PointSuggest/TPointSuggestError';

import noop from '../../lib/noop';
import defaultDataProvider from '../../lib/suggests/defaultDataProvider';
import {reachGoal} from '../../lib/yaMetrika';
import {ceilByStep} from '../../lib/mathUtils';

import Popup from '../Popup/Popup';
import PopupError from '../PopupError/PopupError';
import PointInput from '../PointInput/PointInput';
import PointSuggestDrop from '../PointSuggestDrop/PointSuggestDrop';

const suggestPositions = [[PrimaryPosition.bottom, SecondaryPosition.left]];
const errorPositions = [[PrimaryPosition.bottom, SecondaryPosition.center]];

const b = B('PointSuggest');

interface IPointSuggestProps {
    value: TPointSuggestItem;
    onChange: TPointSuggestCallback;
    id: string;
    name: string;
    placeholder: string;
    errors: TPointSuggestError[];

    dataProvider?: TSuggestDataProvider;
    dataProviderOptions?: ISuggestDataProviderOptions;
    setupSuggests?: (
        items: TSuggestItemFromBackend[],
    ) => TSuggestItemFromBackend[];
    autoFocus?: boolean;
    onFocus?: () => void;
    onApply?: () => void;
}

interface IPointSuggestState {
    items: TSuggestItemFromBackend[];
    selectedItemIndex: number;
    focused: boolean;
    dropHidden: boolean;
    changes: number; // Кол-во введенных символов до выбора из саджестов
    fetchedAt: null | number; // Время получения данных из саджестов
}

/** Старые саджесты переписанные на ts. Не используйте их, пожалуйста. Есть другая версия саджестов для тача и десктопа. */
export default class PointSuggest extends React.PureComponent<
    IPointSuggestProps,
    IPointSuggestState
> {
    state: IPointSuggestState = {
        items: [],
        selectedItemIndex: -1,
        focused: false,
        dropHidden: false,

        // Данные для метрики
        changes: 0, // кол-во введенных символов до выбора из саджестов
        fetchedAt: null, // время получения данных из саджестов
    };

    pointInputRef = React.createRef<PointInput>();

    static defaultProps = {
        dataProvider: defaultDataProvider,
        dataProviderOptions: {
            url: '//suggests.rasp.yandex.net',
            path: 'all_suggests',
            query: {
                format: 'old',
                lang: 'ru',
                client_city: 213,
                national_version: 'ru',
            },
        },
    };

    onKeyDown = (e: React.KeyboardEvent<HTMLInputElement>): void => {
        switch (e.keyCode) {
            case KeyCode.up:
                this.shiftItemSelection(e, -1);
                break;
            case KeyCode.down:
                this.shiftItemSelection(e, 1);
                break;
            case KeyCode.enter:
                this.applySelectedItem(e);
                break;
            case KeyCode.esc:
                this.setState({dropHidden: true});
                break;
        }
    };

    onChange: TPointSuggestCallback = (e, data) => {
        this.setState({
            selectedItemIndex: -1,
            dropHidden: false,
        });

        this.trackChange();
        this.requestItems(data.value.title);
        this.props.onChange(e, data);
    };

    onFocus = (): void => {
        this.setState({
            focused: true,
            dropHidden: false,
        });
        this.requestItems(this.props.value.title);
        this.props.onFocus?.();
    };

    onBlur = (): void => {
        this.setState({focused: false});
    };

    onDropItemClick: TPointSuggestCallback = (e, data) => {
        this.setState({dropHidden: true});
        this.onSuggestPointSelected();
        this.props.onChange(e, data);
    };

    onDropItemMouseOver: TPointSuggestCallback = (e, data) => {
        this.setState({
            selectedItemIndex: data.value.key
                ? this.getItemIndexByKey(data.value.key)
                : -1,
        });
    };

    onSuggestPointSelected = (): void => {
        const {changes, fetchedAt} = this.state;

        reachGoal('point_suggest_selected', {
            changes,
            timeToSelectFromSuggests: fetchedAt
                ? ceilByStep(Date.now() - fetchedAt, 200)
                : null,
        });

        this.setState({
            changes: 0,
            fetchedAt: null,
        });

        if (this.props.onApply) {
            setTimeout(this.props.onApply, 0);
        }
    };

    getItemIndexByKey(key: Point): number {
        const items = this.state.items;

        for (let i = 0; i < items.length; i++) {
            if (key === items[i].value.key) {
                return i;
            }
        }

        return -1;
    }

    shiftItemSelection(
        e: React.KeyboardEvent<HTMLInputElement>,
        shift: number,
    ): void {
        const {items, selectedItemIndex: currentIndex} = this.state;
        const length = items.length;

        e.preventDefault();

        if (length > 0) {
            const selectedItemIndex =
                currentIndex === -1 && shift === -1
                    ? length - 1
                    : (currentIndex + length + shift) % length;

            const value = items[selectedItemIndex].value;

            this.setState({selectedItemIndex});
            this.props.onChange(e, {value});
        }
    }

    applySelectedItem(e: React.KeyboardEvent<HTMLInputElement>): void {
        if (!this.isDropVisible() || this.state.selectedItemIndex === -1) {
            this.pointInputRef.current?.triggerLeaveFocus();

            return;
        }

        const {items, selectedItemIndex} = this.state;

        e.preventDefault();

        if (selectedItemIndex !== -1) {
            this.onSuggestPointSelected();
            this.props.onChange(e, {
                value: items[selectedItemIndex].value,
            });
        }

        this.setState({dropHidden: true, selectedItemIndex: -1});
    }

    requestItems(text: string): void {
        const {dataProvider, setupSuggests, dataProviderOptions} = this.props;

        if (!dataProvider || !dataProviderOptions) {
            return;
        }

        dataProvider(text, dataProviderOptions)
            .then(items => (setupSuggests ? setupSuggests(items) : items))
            .then(items => {
                if (this.props.value.title === text) {
                    this.setState({
                        items,
                        fetchedAt: Date.now(),
                    });
                }
            })
            .catch(noop);
    }

    isDropVisible(): boolean {
        const state = this.state;

        return !state.dropHidden && state.focused && state.items.length > 0;
    }

    trackChange(): void {
        this.setState({
            changes: this.state.changes + 1,
        });
    }

    render(): React.ReactElement {
        const {id, name, value, errors, autoFocus, placeholder} = this.props;
        const {items, selectedItemIndex} = this.state;
        const dropVisible = this.isDropVisible();

        return (
            <div className={b()}>
                <PointInput
                    id={id}
                    value={value}
                    name={name}
                    placeholder={placeholder}
                    onKeyDown={this.onKeyDown}
                    onChange={this.onChange}
                    onFocus={this.onFocus}
                    onBlur={this.onBlur}
                    ref={this.pointInputRef}
                    autoFocus={autoFocus}
                />

                <Popup
                    visible={errors.length > 0 && !dropVisible}
                    contentWidth="full"
                    theme="error"
                    positions={errorPositions}
                >
                    {errors.map(error => (
                        <PopupError htmlFor={id} key={error.key}>
                            {error.title}
                        </PopupError>
                    ))}
                </Popup>

                <Popup
                    visible={dropVisible}
                    positions={suggestPositions}
                    withoutArrow
                >
                    <PointSuggestDrop
                        className={b('suggestList')}
                        items={items}
                        selectedItemIndex={selectedItemIndex}
                        onItemClick={this.onDropItemClick}
                        onItemMouseOver={this.onDropItemMouseOver}
                    />
                </Popup>
            </div>
        );
    }
}
