import {React, PureComponent, B, bindMethods} from '../base';

import SecondaryPosition from '../../interfaces/lib/dimensions/SecondaryPosition';
import PrimaryPosition from '../../interfaces/lib/dimensions/PrimaryPosition';

import noop from '../../lib/noop';
import doNotRenderUntil from '../_utils/doNotRenderUntil';

import getElementRect from '../../lib/dimensions/getElementRect';
import getWindowRect from '../../lib/dimensions/getWindowRect';
import selectPopupPosition from '../../lib/dimensions/selectPopupPosition';
import getRootRect from '../../lib/dimensions/getRootRect';

const b = B('Popup');

export const ANIMATION_DURATION = 200;

const DEFAULT_POSITION = [PrimaryPosition.bottom, SecondaryPosition.center];

class Popup extends PureComponent {
    hackTimeout = null;

    static defaultProps = {
        visible: false,
        onShow: noop,
        contentClassName: '',
    };

    constructor(props) {
        super(props);

        this.state = {
            visible: false,
            animated: false,
        };

        bindMethods(this, ['makeSelfRef', 'makeContentRef', 'onClickOutside']);
    }

    componentDidMount() {
        this.handlePropsTransition(Popup.defaultProps, this.props);
    }

    componentDidUpdate(prevProps) {
        this.handlePropsTransition(prevProps, this.props);
    }

    componentWillUnmount() {
        this.removeEvents();

        clearTimeout(this.hackTimeout);
    }

    onClickOutside(e) {
        if (!this._content.contains(e.target)) {
            this.props.onClickOutside(e);
        }
    }

    handlePropsTransition(prevProps, nextProps) {
        if (prevProps.visible === nextProps.visible) {
            return;
        }

        this.handleWindowEvents(nextProps);

        if (!nextProps.visible) {
            this.setState({visible: nextProps.visible});
            setTimeout(
                () => this.setState({animated: false}),
                ANIMATION_DURATION,
            );

            return;
        }

        this.selectPosition();

        this.setState(
            {
                visible: nextProps.visible,
                animated: true,
            },
            () => this.props.onShow(),
        );

        // Хак для хрома 57
        this.hackTimeout = setTimeout(() => {
            this._content.style.display = 'block';
        }, 0);
    }

    handleWindowEvents(props) {
        if (this.props.visible && props.onClickOutside) {
            this.addEvents();
        } else {
            this.removeEvents();
        }
    }

    makeSelfRef(node) {
        this._self = node;
    }

    makeContentRef(node) {
        this._content = node;
    }

    addEvents() {
        document.addEventListener('click', this.onClickOutside);
    }

    removeEvents() {
        document.removeEventListener('click', this.onClickOutside);
    }

    getDimensions() {
        return this._content.getBoundingClientRect();
    }

    getParentRect() {
        return getElementRect(this._self.parentNode);
    }

    selectPosition() {
        const {visible, positions} = this.props;

        if (!positions || !visible) {
            return;
        }

        const dimensions = this.getDimensions();
        const parentRect = this.getParentRect();
        const windowRect = getWindowRect();
        const rootRect = getRootRect();

        try {
            const position = selectPopupPosition({
                dimensions,
                parentRect,
                windowRect,
                rootRect,
                positions,
            });

            this.setState({position});
        } catch (e) {
            this.setState({position: DEFAULT_POSITION});
        }
    }

    render() {
        const {
            style,
            children,
            className,
            arrowClassName,
            contentClassName,
            getContent,
            withoutArrow,
            contentWidth,
            minContentWidth,
            theme,
        } = this.props;
        const {visible, animated} = this.state;
        const position =
            this.state && this.state.position && this.state.position.join('-');

        return (
            <div
                className={b(
                    {
                        contentWidth,
                        minContentWidth,
                        visible,
                        position,
                        animated,
                        theme,
                        withoutArrow,
                    },
                    className,
                )}
                style={style}
                ref={this.makeSelfRef}
            >
                <span className={b('arrow', arrowClassName)} />

                <div
                    className={b('content', contentClassName)}
                    ref={this.makeContentRef}
                >
                    {getContent ? getContent() : children}
                </div>
            </div>
        );
    }
}

export default doNotRenderUntil(Popup);
