import {React, Component, B, refCallback, connect} from '../base';

import Helmet from 'react-helmet';
import Loadable from 'react-loadable';
import {momentTimezone as moment} from '../../../reexports';
import isEqual from 'lodash/isEqual';
import pick from 'lodash/pick';

import isFetching from '../../lib/isFetching';
import getMetaInformation from '../../lib/meta/metaTags';
import {shouldShowSearchDescription} from '../../lib/seo/search';
import scrollWindow from '../../../client/lib/scrollWindow';
import {getThreadBreadCrumbs} from '../../lib/thread/thread';
import getStationBreadcrumbs from '../../lib/station/getStationBreadcrumbs';
import getSearchBreadCrumbs from '../../lib/url/crumble/getCrumbs';
import getInfoBreadCrumbs from '../../lib/info/getInfoBreadCrumbs';

import {SEARCH, SEARCH_FORM_PAGE_NAME} from '../../routes/search';
import {HOME_PAGE_NAME} from '../../routes';
import {PAGE_404} from '../../routes/page404';
import {THREAD_PAGE_NAME} from '../../routes/thread';
import {TRANSPORT_CITY_PAGE_NAME} from '../../routes/transportCity';
import {TRANSPORT_PAGE_NAME} from '../../routes/transport';
import {STATION_PAGE_NAME} from '../../routes/station';
import {INFO_PAGE_NAME} from '../../routes/info';
import {AMBIGUOUS_PAGE_NAME} from '../../components/AmbiguousPage/AmbiguousPage';

import {RootContextProvider} from './RootContext';
import GlobalPopupContext from './GlobalPopupContext';
import Header from '../Header/Header';
import Footer from '../basic/Footer';
import Tooltip from '../Tooltip/Tooltip';
import HomePage from '../HomePage';
import SearchPage from '../SearchPage';
import InfoPage from '../InfoPage/InfoPage';
import ErrorPage from '../ErrorPage/ErrorPage';
import FooterInformation from '../basic/FooterInformation';
import Breadcrumbs from '../Breadcrumbs';
import Error404Block from '../Error404Block';
import LogoAndPersonal from '../Header/LogoAndPersonal';
import LoadingChunk from '../basic/LoadingChunk';

import HomeLayout from '../layout/HomeLayout';
import SearchLayout from '../layout/SearchLayout/SearchLayout';
import ThreadLayout from '../layout/ThreadLayout';
import TransportCityLayout from '../layout/TransportCityLayout';
import TransportLayout from '../layout/TransportLayout';
import StationLayout from '../layout/StationLayout';
import InfoLayout from '../layout/InfoLayout';
import Direct from '../Direct/Direct';
import HeaderDirectBlock from '../HeaderDirectBlock/HeaderDirectBlock';

const b = B('Root');

const isMobile = process.env.PLATFORM === 'mobile';

const BugReporter = Loadable({
    // перед рефакторингом загляни в server/template/getCssLoadableChunksByPageType.ts
    loader: () => import('../BugReporter/BugReporter'),
    loading() {
        return null;
    },
});

const ThreadPage = Loadable({
    // перед рефакторингом загляни в server/template/getCssLoadableChunksByPageType.ts
    loader: () => import('../ThreadPage'),
    loading: LoadingChunk,
});

const TransportCityPage = Loadable({
    // перед рефакторингом загляни в server/template/getCssLoadableChunksByPageType.ts
    loader: () => import('../TransportCityPage'),
    loading: LoadingChunk,
});

const TransportPage = Loadable({
    // перед рефакторингом загляни в server/template/getCssLoadableChunksByPageType.ts
    loader: () => import('../TransportPage'),
    loading: LoadingChunk,
});

const StationPage = Loadable({
    // перед рефакторингом загляни в server/template/getCssLoadableChunksByPageType.ts
    loader: () => import('../StationPage'),
    loading: LoadingChunk,
});

class Root extends Component {
    constructor(props) {
        super(props);

        // Делает экран неподвижным (нельзя посколлить)
        this.fixBody = () => {
            this.setState({
                bodyFixed: true,
                scrollTop: window.pageYOffset,
            });
        };

        // Отменяет неподвижность экрана (противоположность fixBody)
        this.releaseBody = () => {
            let scrollTop;

            if (this.state.bodyFixed) {
                this.setState(
                    oldState => {
                        scrollTop = oldState.scrollTop;

                        return {
                            bodyFixed: false,
                            scrollTop: undefined,
                        };
                    },
                    () => {
                        window.scrollTo(0, scrollTop);
                    },
                );
            }
        };

        this.state = {
            bodyFixed: false,
            scrollTop: undefined,
            headerIsOpened: false,
            popupIsOpened: false,
            /* eslint-disable react/no-unused-state */
            fixBody: this.fixBody,
            releaseBody: this.releaseBody,
            /* eslint-enable react/no-unused-state */
        };

        this.globalPopupContext = {
            openGlobalPopup: this.openPopup,
            closeGlobalPopup: this.closePopup,
        };
    }

    componentDidUpdate(prevProps, prevState) {
        const {popupIsOpened: prevPopupIsOpened} = prevState;
        const {popupIsOpened: currentPopupIsOpened} = this.state;
        const popupToggled = prevPopupIsOpened !== currentPopupIsOpened;

        if (!popupToggled) {
            return;
        }

        if (currentPopupIsOpened) {
            window.scrollTo(0, 0);
        } else {
            window.scrollTo(0, this.state.scrollTop + 1);
            this.setState({
                scrollTop: undefined,
            });
        }
    }

    onHeaderOpen = () => {
        this.setState({
            headerIsOpened: true,
        });
        this.fixBody();
    };

    onHeaderClose = () => {
        this.setState({
            headerIsOpened: false,
        });
        this.releaseBody();
    };

    getMeta() {
        const {state} = this.props;

        // Заголовок не должен меняться / мигать, пока страница подкачивает данные
        if (state.page.fetching === null || !this.metaCache) {
            this.metaCache = getMetaInformation(state);
        }

        return this.metaCache;
    }

    scrollTo = scroll => {
        const headerHeight = this.header.getHeightAt(scroll);

        scrollWindow(scroll - headerHeight);
    };

    closePopup = () => {
        this.setState({
            popupIsOpened: false,
        });
    };

    openPopup = () => {
        const popupJustOpened = !this.state.popupIsOpened;

        const newState = {
            popupIsOpened: true,
            scrollTop: popupJustOpened
                ? window.pageYOffset
                : this.state.scrollTop,
        };

        const oldState = pick(this.state, Object.keys(newState));

        if (!isEqual(newState, oldState)) {
            this.setState(newState);
        }
    };

    getContent(pageType) {
        const {state} = this.props;

        const {
            tld,
            searchForm,
            currentSettlement,
            clientSettlement,
            directions,
            popularDirections,
            suggests,
            nationalVersion,
            teasers,
            stationsGroup,
            page,
            search,
            flags,
            station,
            transport,
            transportCity,
            home,
        } = state;

        switch (pageType) {
            case HOME_PAGE_NAME:
                return (
                    <HomeLayout page={page}>
                        <HomePage
                            page={page}
                            tld={tld}
                            clientSettlement={clientSettlement}
                            currentSettlement={currentSettlement}
                            directions={directions}
                            popularDirections={popularDirections}
                            suggests={suggests}
                            nationalVersion={nationalVersion}
                            stationsGroup={stationsGroup}
                            teasers={teasers}
                            home={home}
                            flags={flags}
                        />
                    </HomeLayout>
                );
            case SEARCH: {
                const hasDescription = shouldShowSearchDescription(
                    search,
                    page,
                );

                return (
                    <>
                        <SearchLayout
                            search={search}
                            isFetching={isFetching(search, page, flags)}
                            hasDescription={hasDescription}
                            flags={flags}
                        >
                            <SearchPage />
                        </SearchLayout>
                    </>
                );
            }

            case THREAD_PAGE_NAME:
                return (
                    <ThreadLayout flags={flags}>
                        <ThreadPage />
                    </ThreadLayout>
                );
            case TRANSPORT_CITY_PAGE_NAME:
                return (
                    <TransportCityLayout>
                        <TransportCityPage {...transportCity} page={page} />
                    </TransportCityLayout>
                );
            case TRANSPORT_PAGE_NAME:
                return (
                    <TransportLayout>
                        <TransportPage {...transport} page={page} />
                    </TransportLayout>
                );
            case STATION_PAGE_NAME:
                return (
                    <StationLayout subtype={station.currentSubtype}>
                        <StationPage />
                    </StationLayout>
                );
            case INFO_PAGE_NAME:
                return (
                    <InfoLayout>
                        <InfoPage />
                    </InfoLayout>
                );
            case PAGE_404: {
                return isMobile ? (
                    <Error404Block />
                ) : (
                    <Direct className={b('direct404')} blockId="R-I-94177-42" />
                );
            }

            default:
                return (
                    <SearchLayout search={search} flags={flags}>
                        <ErrorPage
                            type={page.current}
                            searchForm={searchForm}
                            search={search}
                        />
                    </SearchLayout>
                );
        }
    }

    getPageType() {
        const {page} = this.props.state;

        return page.fetching || page.current;
    }

    getLayoutType() {
        return isMobile ? 'touch' : 'desktop';
    }

    renderBreadCrumbs() {
        const {
            page,
            search,
            thread: {title, stations, transportType},
            station: {
                id,
                title: stationTitle,
                type,
                currentSubtype,
                mainSubtype,
                event,
                settlement,
                cityData,
                terminalName,
                terminals,
            },
            info,
            tld,
            language,
            flags,
            platform,
        } = this.props.state;

        const layout = this.getLayoutType();
        const pageType = this.getPageType();

        let crumbs = [];

        if (page.current === SEARCH && page.fetching === null) {
            crumbs = getSearchBreadCrumbs(
                search.context,
                tld,
                language,
                page,
                search.filtering,
            );
        }

        if (
            page.current === THREAD_PAGE_NAME &&
            page.fetching === null &&
            platform === 'desktop'
        ) {
            crumbs = getThreadBreadCrumbs({
                title,
                stations,
                transportType,
                tld,
                language,
                flags,
            });
        }

        if (page.current === STATION_PAGE_NAME && page.fetching === null) {
            const {cityStations} = cityData || {};
            const terminal = terminals.find(
                terminalObj => terminalObj.name === terminalName,
            );

            crumbs = getStationBreadcrumbs(
                id,
                stationTitle,
                type,
                tld,
                language,
                flags,
                currentSubtype,
                mainSubtype,
                settlement,
                event,
                cityStations,
                terminal,
            );
        }

        if (page.current === INFO_PAGE_NAME && page.fetching === null) {
            crumbs = getInfoBreadCrumbs({
                title: info.title,
                parents: info.parents,
                tld,
                language,
            });
        }

        return (
            <Breadcrumbs
                elements={crumbs}
                layout={layout}
                pageType={pageType}
            />
        );
    }

    render() {
        const {state, throwError, dispatch} = this.props;
        const {bodyFixed, headerIsOpened, popupIsOpened} = this.state;

        const {
            now,
            tld,
            language,
            nationalVersion,
            searchForm,
            searchFormHints,
            page,
            user,
            search,
            clientSettlement,
            currentSettlement,
            tooltip,
            environment,
            suggests,
            seoQueryParams,
            home,
            transportCity,
            transport,
            teasers,
        } = state;

        const showBugReporter = user.isStaff;

        const layout = this.getLayoutType();

        const pageType = this.getPageType();
        const isHomePage = pageType === HOME_PAGE_NAME;
        const isThreadPage = pageType === THREAD_PAGE_NAME;
        const isStationPage = pageType === STATION_PAGE_NAME;
        const isTransportCityPage = pageType === TRANSPORT_CITY_PAGE_NAME;
        const isTransportPage = pageType === TRANSPORT_PAGE_NAME;
        const isInfoPage = pageType === INFO_PAGE_NAME;
        const isAmbiguousPage = pageType === AMBIGUOUS_PAGE_NAME;
        const isSearchFormPage = pageType === SEARCH_FORM_PAGE_NAME;
        const is404Page = pageType === PAGE_404;
        const needDisplayHeader = !(
            isMobile &&
            (isThreadPage || isStationPage || isInfoPage)
        );

        const displayLogoAndPersonalIndividually =
            isMobile &&
            (isHomePage ||
                isThreadPage ||
                isTransportCityPage ||
                isTransportPage ||
                isStationPage ||
                isInfoPage);
        const needDisplayFooterInformation =
            !is404Page &&
            !isThreadPage &&
            !isStationPage &&
            !(isMobile && isAmbiguousPage) &&
            !(isMobile && isSearchFormPage) &&
            !(isMobile && isInfoPage);

        const main = (
            <main className={b('main')}>{this.getContent(pageType)}</main>
        );

        return (
            <RootContextProvider value={this.state}>
                <GlobalPopupContext.Provider value={this.globalPopupContext}>
                    <div
                        className={b({
                            layout,
                            pageType,
                            fixed: bodyFixed,
                            popupIsOpened,
                        })}
                    >
                        <div className={b('popup')} id="globalPopupContainer" />

                        <div
                            className={b('mainContent')}
                            style={{
                                top: bodyFixed ? -1 * this.state.scrollTop : 0,
                            }}
                        >
                            <Helmet {...this.getMeta()} />

                            <HeaderDirectBlock
                                pageType={pageType}
                                isMobile={isMobile}
                                className={b('nativeDirectInHeaderTop')}
                            />

                            {displayLogoAndPersonalIndividually && (
                                <LogoAndPersonal
                                    tld={tld}
                                    language={language}
                                    user={user}
                                    page={page}
                                    clientSettlement={clientSettlement}
                                    className={b('logoAndPersonal', {
                                        responsive: isThreadPage,
                                    })}
                                />
                            )}

                            {needDisplayHeader && (
                                <Header
                                    ref={refCallback(this, 'header')}
                                    tld={tld}
                                    user={user}
                                    dispatch={dispatch}
                                    searchForm={searchForm}
                                    searchFormHints={searchFormHints}
                                    language={language}
                                    suggests={suggests}
                                    clientSettlement={clientSettlement}
                                    currentSettlement={currentSettlement}
                                    transportCity={transportCity}
                                    transport={transport}
                                    home={home}
                                    environment={environment}
                                    page={page}
                                    nationalVersion={nationalVersion}
                                    pageType={pageType}
                                    showOnlyOpenedForm={
                                        displayLogoAndPersonalIndividually
                                    }
                                    // mobile header props
                                    opened={headerIsOpened}
                                    onOpen={this.onHeaderOpen}
                                    onClose={this.onHeaderClose}
                                    search={search}
                                    teasers={isHomePage ? teasers : undefined}
                                    renderBanner={isHomePage}
                                />
                            )}

                            {main}

                            {this.renderBreadCrumbs()}

                            {needDisplayFooterInformation && (
                                <FooterInformation
                                    page={page}
                                    tld={tld}
                                    isProduction={environment.production}
                                    pageType={pageType}
                                    seoQueryParams={seoQueryParams}
                                    clientId={user.clientId}
                                />
                            )}

                            <Footer
                                tld={tld}
                                now={now}
                                user={user}
                                page={page}
                                timezone={clientSettlement.timezone}
                                throwError={throwError}
                                layout={layout}
                                pageType={pageType}
                                language={language}
                            />

                            <Tooltip dispatch={dispatch} {...tooltip} />
                            {showBugReporter && <BugReporter />}
                        </div>
                    </div>
                </GlobalPopupContext.Provider>
            </RootContextProvider>
        );
    }
}

const mapStateToProps = state => ({state});

const connected = connect(mapStateToProps)(Root);

export {connected as Root, Helmet, Loadable, moment};
