import {React, PureComponent, B, bindMethods} from '../base';

import debounce from 'lodash/debounce';

import {ROBOT} from '../../lib/date/formats';
import {getToday, getRangeFromDate} from '../../lib/date/utils';
import {getCommonMask, getMonthsByMask} from '../../lib/date/mask';
import {reachGoalOnce} from '../../lib/yaMetrika';

import Arrow from '../Arrow/Arrow';
import Month from '../Month/Month';
import Button from '../Button/Button';
import Weekdays from '../Weekdays/Weekdays';
import SearchCalendarDayLink from '../SearchCalendarDayLink/SearchCalendarDayLink';

const b = B('SearchCalendarSlider');

const DEBOUNCE_TIME = 100;
const WINDOW_WIDTH_BREAKPOINT = 1340;
const MONTHS_PER_PAGE = {
    NARROW: 3,
    WIDE: 4,
};

export default class SearchCalendarSlider extends PureComponent {
    constructor(props) {
        super(props);

        this.state = {
            position: 0,
            full: false,
            monthsPerPage: MONTHS_PER_PAGE.NARROW,
        };

        bindMethods(this, ['onBackwardFlipperClick', 'onForwardFlipperClick']);

        this.onWindowResize = debounce(
            this.recalcMonthsPerPage,
            DEBOUNCE_TIME,
        ).bind(this);
    }

    componentDidMount() {
        this.recalcMonthsPerPage();
        window.addEventListener('resize', this.onWindowResize);
    }

    componentWillUnmount() {
        window.removeEventListener('resize', this.onWindowResize);
    }

    onBackwardFlipperClick() {
        this.shiftBy(-1);
        reachGoalOnce('search_calendar_flip_backward');
    }

    onForwardFlipperClick() {
        this.shiftBy(1);
        reachGoalOnce('search_calendar_flip_forward');
    }

    onDayClick() {
        reachGoalOnce('search_calendar_day_click');
    }

    recalcMonthsPerPage() {
        const winWidth = window.innerWidth;

        if (
            winWidth > WINDOW_WIDTH_BREAKPOINT &&
            this._months.length > MONTHS_PER_PAGE.NARROW
        ) {
            this.setState({
                monthsPerPage: MONTHS_PER_PAGE.WIDE,
                position: Math.min(
                    this.state.position,
                    this._months.length - MONTHS_PER_PAGE.WIDE,
                ),
            });
        } else {
            this.setState({monthsPerPage: 3});
        }
    }

    shiftBy(shift) {
        const {position} = this.state;

        this.setState({
            full: true,
            position: position + shift,
        });
    }

    render() {
        const {segments, context, language, ...props} = this.props;

        const {full, position, monthsPerPage} = this.state;

        const today = getToday(context.time);
        const range = getRangeFromDate(today);
        const mask = getCommonMask(segments);
        const months = getMonthsByMask({mask, timezone: context.time.timezone});
        const monthsToRender = full ? months : months.slice(0, monthsPerPage);

        this._months = months;

        const showFlippers = months.length > monthsPerPage;

        return (
            <div className={b({monthsPerPage})}>
                <div
                    className={b('months')}
                    style={{
                        transform: `translateX(${-100 * position}%)`,
                    }}
                >
                    {monthsToRender.map((month, i) => (
                        <div className={b('month')} key={month.format(ROBOT)}>
                            <Weekdays
                                className={b('weekdays')}
                                language={language}
                            />
                            <Month
                                mask={mask}
                                month={month}
                                today={today}
                                range={range}
                                context={context}
                                language={language}
                                onClick={this.onDayClick}
                                DayComponent={SearchCalendarDayLink}
                                showYear={i === 0 || month.month() === 0} // Первый месяц в маске или январь
                                {...props}
                            />
                        </div>
                    ))}
                </div>

                {showFlippers && (
                    <Button
                        className={b('flipper', {direction: 'backward'})}
                        onClick={this.onBackwardFlipperClick}
                        disabled={position <= 0}
                    >
                        <Arrow direction="left" />
                    </Button>
                )}

                {showFlippers && (
                    <Button
                        className={b('flipper', {direction: 'forward'})}
                        onClick={this.onForwardFlipperClick}
                        disabled={position >= months.length - monthsPerPage}
                    >
                        <Arrow direction="right" />
                    </Button>
                )}
            </div>
        );
    }
}
