import React, {ReactElement, memo, useEffect} from 'react';
import B from 'bem-cn-lite';

import ISearchContext from '../../../common/interfaces/state/search/ISearchContext';
import ISearchSort from '../../../common/interfaces/state/search/ISearchSort';
import IStateSearchForm from '../../../common/interfaces/state/IStateSearchForm';
import SearchErrorPageType from '../../../common/interfaces/components/searchErrorPage/SearchErrorPageType';
import ISearchArchivalData from '../../../common/interfaces/state/search/ISearchArchivalData';

import {reachGoal} from '../../lib/yaMetrika';
import notFoundTitle from '../../lib/meta/notFoundTitle';

import getSearchErrorField from '../../../common/lib/meta/getSearchErrorField';
import getSearchFormErrorInfo from '../../../common/lib/meta/getSearchFormErrorInfo';

const b = B('SearchErrorBase');

interface ISearchErrorBaseProps {
    context: ISearchContext;
    type: SearchErrorPageType;

    sort?: ISearchSort;
    searchForm?: IStateSearchForm;
    archivalData?: ISearchArchivalData;
    isMobile?: boolean;
}

export default memo(SearchErrorBase);

function SearchErrorBase({
    context,
    type,

    sort,
    searchForm,
    archivalData,
    isMobile,
}: ISearchErrorBaseProps): ReactElement {
    const searchFormErrorInfo = searchForm
        ? getSearchFormErrorInfo(searchForm)
        : null;
    const typeOfSearchError = searchFormErrorInfo
        ? searchFormErrorInfo.errorType
        : type;

    useEffect(() => {
        if (searchForm && typeOfSearchError === SearchErrorPageType.reload) {
            const {errors} = searchForm;

            if (errors) {
                reachGoal('search_error_page_load', {
                    errorType: errors.map(
                        error => `${getSearchErrorField(error)}-${error.type}`,
                    ),
                });
            }
        }

        if (typeOfSearchError === SearchErrorPageType.noTickets) {
            reachGoal('no_routes_hints_page_load', {
                context: {
                    from: context.userInput.from,
                    to: context.userInput.to,
                    transportType: context.transportType,
                    date: context.when.date,
                },
            });
        }
    }, [context, searchForm, typeOfSearchError]);

    const {title, subtitle} = notFoundTitle({context, sort, archivalData});

    const {title: archivalDataTitle, subtitle: archivalDataSubtitle} =
        archivalData
            ? notFoundTitle({context, sort})
            : {title: '', subtitle: ''};

    const reloadTitle = searchFormErrorInfo ? searchFormErrorInfo.title : null;
    const reloadSubtitle = searchFormErrorInfo
        ? searchFormErrorInfo.subtitle
        : null;

    return (
        <div className={b({isMobile})}>
            {typeOfSearchError === SearchErrorPageType.reload ? (
                <img
                    className={b('icon')}
                    src={`${process.env.IMAGES_PATH}/reload.svg`}
                    alt="reload-icon"
                />
            ) : (
                <img
                    className={b('icon')}
                    src={`${process.env.IMAGES_PATH}/noTickets_v2.svg`}
                    alt="no-tickets-icon"
                />
            )}

            {reloadTitle ? (
                <div className={b('title')}>{reloadTitle}</div>
            ) : (
                <h1 className={b('title')}>{title}</h1>
            )}

            {typeOfSearchError === SearchErrorPageType.noTickets ? (
                <p className={b('text')}>{reloadSubtitle || subtitle}</p>
            ) : null}

            {archivalData && (
                <>
                    <p className={b('text')}>{`${archivalDataTitle}.`}</p>

                    <p className={b('text')}>{archivalDataSubtitle}</p>
                </>
            )}
        </div>
    );
}
