import React from 'react';
import B from 'bem-cn-lite';

import {HUMAN} from '../../lib/date/formats';

import Tld from '../../interfaces/Tld';
import {FilterTransportType} from '../../lib/transportType';
import {IWithClassName} from '../../interfaces/components/IWithClassName';

import {getLastYearPeriod, getToday} from '../../lib/date/utils';
import {belarusSearchDisclaimerIsAvailable} from '../../lib/disclaimers/disclaimersAvailability';
import isLastochkaPage from '../../lib/search/isLastochkaPage';
import useSelector from '../useSelector';
import insertMarkupIntoKey from '../../i18n/_utils/insertMarkupIntoKey.js';

import HelpLinks from '../HelpLinks/HelpLinks';
import BaseDisclaimer from '../Disclaimers/BaseDisclaimer';
import AttentionDisclaimer from '../Disclaimers/AttentionDisclaimer';
import DynamicPricingDisclaimer from '../Disclaimers/DynamicPricingDisclaimer';
import ElectronicTicketDisclaimer from '../Disclaimers/ElectronicTicketDisclaimer';

import disclaimersKeyset from '../../i18n/disclaimers';

interface ISearchFooterProps extends IWithClassName {
    withHelpLinks?: boolean;
}

const b = B('SearchFooter');

const SearchFooter: React.FC<ISearchFooterProps> = ({
    className,
    withHelpLinks,
}) => {
    const sort = useSelector(state => state.search.sort);
    const filtering = useSelector(state => state.search.filtering);
    const stats = useSelector(state => state.search.stats);
    const context = useSelector(state => state.search.context);
    const isBusSearchResult = useSelector(
        state => state.search.isBusSearchResult,
    );
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const currencies = useSelector(state => state.currencies);

    const {transportType, when} = context;
    const {nationalCurrency, preferredCurrency} = currencies;

    const national = tld === Tld.ua ? Tld.ua : Tld.ru;

    const isSuburban = transportType === FilterTransportType.suburban;
    const isBusOnly = isBusSearchResult;

    const eTicketDisclaimer = stats.hasElectronicTicket;
    const dynamicPricingDisclaimer = stats.hasDynamicPricing;

    const currencyDisclaimer =
        !isSuburban && nationalCurrency !== preferredCurrency;
    const scheduleParams = {
        from: context.from.title,
        to: context.to.title,
        years: getLastYearPeriod(context.time),
        today: (
            <span data-nosnippet key="schedule">
                {getToday(context.time).format(HUMAN)}
            </span>
        ),
    };

    const currencyDisclaimerKey = `currency-${national}-${
        isBusOnly ? FilterTransportType.bus : FilterTransportType.all
    }`;

    return (
        <div className={b({}, className)}>
            {withHelpLinks && (
                <HelpLinks
                    sort={sort}
                    context={context}
                    filtering={filtering}
                    currencies={currencies}
                />
            )}

            <div className={b('disclaimers')}>
                {eTicketDisclaimer && (
                    <ElectronicTicketDisclaimer
                        language={language}
                        type="electronicDisclaimer"
                    />
                )}

                {process.env.PLATFORM === 'mobile' &&
                    dynamicPricingDisclaimer && (
                        <DynamicPricingDisclaimer
                            language={language}
                            type="dynamicPricing"
                        />
                    )}

                {currencyDisclaimer && (
                    <BaseDisclaimer
                        content={disclaimersKeyset(currencyDisclaimerKey)}
                        type={currencyDisclaimerKey}
                    />
                )}

                <BaseDisclaimer
                    content={disclaimersKeyset('pricing-all')}
                    type="pricing-all"
                />

                {isSuburban && (
                    <BaseDisclaimer
                        content={disclaimersKeyset('local-time')}
                        type="local-time"
                    />
                )}

                {isSuburban && (
                    <BaseDisclaimer
                        content={insertMarkupIntoKey(
                            disclaimersKeyset,
                            isLastochkaPage(transportType, when, filtering)
                                ? 'midnight-schedule-lastochka'
                                : 'midnight-schedule',
                            scheduleParams,
                        )}
                        type="midnight-schedule"
                    />
                )}

                {isSuburban && (
                    <AttentionDisclaimer
                        language={language}
                        content={disclaimersKeyset('summer-corrections')}
                        type="summer-corrections"
                    />
                )}

                {belarusSearchDisclaimerIsAvailable(context, tld) && (
                    <BaseDisclaimer
                        content={disclaimersKeyset('belarus-search')}
                        type="belarus-search"
                    />
                )}
            </div>
        </div>
    );
};

export default React.memo(SearchFooter);
