import {React, B, bindMethods, PureComponent, refCallback} from '../base';

import {connect} from 'react-redux';

import {SEARCH} from '../../routes/search';

import IconGlyph from '../../interfaces/components/IconGlyph';

import {
    setWhen,
    swap,
    submitForm,
    setFromPointFromUser,
    setToPointFromUser,
} from '../../actions/searchForm';
import {reachGoal} from '../../lib/yaMetrika';
import {getTodayPhrase} from '../../lib/date/values';
import {bindWithCache} from '../../lib/cache';
import buildSearchFormSuggestDataProviderOptions from '../../lib/suggests/buildSearchFormSuggestDataProviderOptions';
import getNationalLanguage from '../../lib/lang/getNationalLanguage';

import Icon from '../Icon/Icon';
import Button from '../Button/Button';
import Spinner from '../Spinner/Spinner';
import DatePicker from '../DatePicker/DatePicker';
import PointSuggest from '../PointSuggest/PointSuggest';
import SearchFormHints from '../SearchFormHints/SearchFormHints';

import keyset from '../../i18n/search-form';

const b = B('SearchForm');

const pickErrors = (errors, field, errorKeyset) =>
    errors
        .filter(
            error =>
                error.fields &&
                error.fields.length > 0 &&
                error.fields[error.fields.length - 1] === field,
        )
        .map(error => ({
            ...error,
            title: errorKeyset(`error-${field}-${error.type}`),
        }));

const mapStateToProps = ({generatedInputName, language, flags, page}) => ({
    generatedInputName,
    language,
    flags,
    page,
});

class SearchForm extends PureComponent {
    constructor(props) {
        super(props);
        bindMethods(this, [
            'onFocus',
            'onSubmit',
            'onToChange',
            'makeFormRef',
            'onSwapClick',
            'onFromChange',
            'onWhenChange',
        ]);
    }

    onFromChange(e, data) {
        this.props.dispatch(setFromPointFromUser(data.value));
    }

    onToChange(e, data) {
        this.props.dispatch(setToPointFromUser(data.value));
    }

    onWhenChange(e, data) {
        this.props.dispatch(setWhen(data.value));
    }

    onSwapClick() {
        reachGoal('point_point_swap_button_click');
        this.props.dispatch(swap());
    }

    onFocus(e) {
        const target = e.target;

        if (target !== this._submitButton && target !== this._swapButton) {
            if (target.value && !this.props.focused) {
                setTimeout(() => {
                    // В сафари при смене фокуса инпутов с помощью тачпада иногда фокус выставляется на нужный инпут,
                    // затем быстро меняется на предыдущий и снова на тот, на который нужно.
                    // setSelectionRange в добавку еще ассинхронно вызывает дополнительные события фокуса
                    // и процесс на вечность зацикливается
                    // https://github.com/facebook/react/issues/10871
                    if (document.activeElement !== target) {
                        return;
                    }

                    target.setSelectionRange(0, target.value.length);
                }, 0);
            }

            if (this.props.onFocus) {
                this.props.onFocus();
            }
        }
    }

    onSubmit(e) {
        if (this.props.onSubmit) {
            this.props.onSubmit(e);
        }

        this._form.blur();
        reachGoal('point_point_search_button_click');

        if (e.defaultPrevented) {
            return;
        }

        e.preventDefault();
        this.props.dispatch(submitForm());
    }

    isSearchPageFetching() {
        return this.props.page.fetching === SEARCH;
    }

    makeFormRef(node) {
        this._form = node;
    }

    isDisabled() {
        const {errors, validation} = this.props;

        return (
            errors.length > 0 ||
            this.isSearchPageFetching() ||
            validation !== null
        );
    }

    render() {
        const {
            userInput,
            when,
            errors,
            tld,
            time,
            searchFormHints,
            language,
            suggests,
            environment,
            dispatch,
            flags,
            additionalHiddenInputs,
            setupSuggests,
            submitButtonText,
            focusField,
            nationalVersion,
            clientSettlement,
            transportType,
            generatedInputName,
            showHints,
            page,
            validation,
        } = this.props;
        const pointSuggestFromTitle = {
            title: `from_${generatedInputName}`,
            key: 'fromId',
        };
        const pointSuggestToTitle = {
            title: `to_${generatedInputName}`,
            key: 'toId',
        };

        const {from, to} = userInput;

        const separator = <div className={b('separator')} />;
        const defaultDatepickerValue = getTodayPhrase(language);

        const [fromDataProviderOptions, toDataProviderOptions] = [
            'from',
            'to',
        ].map(field =>
            buildSearchFormSuggestDataProviderOptions({
                field,
                suggests,
                language,
                nationalVersion,
                clientSettlement,
                from,
                to,
                transportType,
            }),
        );

        return (
            <div className={b()}>
                <form
                    className={b('holder')}
                    method="get"
                    onFocus={this.onFocus}
                    onSubmit={this.onSubmit}
                    ref={this.makeFormRef}
                >
                    <PointSuggest
                        id="from"
                        name={pointSuggestFromTitle.title}
                        value={from}
                        placeholder={keyset('label-from')}
                        onChange={this.onFromChange}
                        errors={pickErrors(errors, 'from', keyset)}
                        suggests={suggests}
                        setupSuggests={bindWithCache(
                            setupSuggests,
                            this,
                            'from',
                        )}
                        dataProviderOptions={fromDataProviderOptions}
                        autoFocus={focusField === pointSuggestFromTitle.title}
                    />

                    {separator}

                    <Button
                        className={b('swap')}
                        onClick={this.onSwapClick}
                        refCallback={refCallback(this, '_swapButton')}
                        tabIndex="-1"
                    >
                        <Icon
                            className={b('swapLeft')}
                            glyph={IconGlyph.swapArrow}
                        />
                        <Icon
                            className={b('swapRight')}
                            glyph={IconGlyph.swapArrow}
                        />
                    </Button>

                    {separator}

                    <PointSuggest
                        id="to"
                        name={pointSuggestToTitle.title}
                        value={to}
                        placeholder={keyset('label-to')}
                        onChange={this.onToChange}
                        errors={pickErrors(errors, 'to', keyset)}
                        setupSuggests={bindWithCache(setupSuggests, this, 'to')}
                        dataProviderOptions={toDataProviderOptions}
                        autoFocus={focusField === pointSuggestToTitle.title}
                    />

                    <input
                        type="hidden"
                        name={pointSuggestFromTitle.key}
                        value={from.key}
                    />

                    <input
                        type="hidden"
                        name={pointSuggestToTitle.key}
                        value={to.key}
                    />

                    {separator}

                    <DatePicker
                        id="when"
                        name="when"
                        value={when}
                        defaultValue={defaultDatepickerValue}
                        tld={tld}
                        time={time}
                        language={language}
                        environment={environment}
                        placeholder={keyset('label-when')}
                        onChange={this.onWhenChange}
                        errors={pickErrors(errors, 'when', keyset)}
                        flags={flags}
                        isRedesigned
                        autoFocus={focusField === 'when'}
                    />

                    {language !== getNationalLanguage(tld) ? (
                        <input type="hidden" name="lang" value={language} />
                    ) : null}

                    {additionalHiddenInputs &&
                        additionalHiddenInputs.map(({name, value}, index) => (
                            <input
                                key={index}
                                type="hidden"
                                name={name}
                                value={value}
                            />
                        ))}

                    <Button
                        type="submit"
                        className={b('submit')}
                        disabled={this.isDisabled()}
                        refCallback={refCallback(this, '_submitButton')}
                    >
                        {((validation !== null ||
                            this.isSearchPageFetching()) && (
                            <div className={b('spinnerContainer')}>
                                <Spinner
                                    className={b('spinner')}
                                    size={
                                        page.current === 'home' &&
                                        page.fetching === null
                                            ? 'small'
                                            : 'tiny'
                                    }
                                    color="gray"
                                />
                            </div>
                        )) ||
                            submitButtonText ||
                            keyset('find')}
                    </Button>
                </form>

                {showHints && (
                    <SearchFormHints
                        dispatch={dispatch}
                        searchFormHints={searchFormHints}
                        language={language}
                        time={time}
                        transportType={transportType}
                    />
                )}
            </div>
        );
    }
}

SearchForm.defaultProps = {
    showHints: true,
};

export default connect(mapStateToProps)(SearchForm);
