import {refCallback} from '../base';

import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';
import {connect} from 'react-redux';

import {CHAR_EM_DASH, CHAR_NBSP} from '../../lib/stringUtils';

import KeyCode from '../../interfaces/KeyCode';
import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
import IconGlyph from '../../interfaces/components/IconGlyph';

import {reachGoal} from '../../lib/yaMetrika';
import noop from '../../lib/noop';
import {buildSpecial} from '../../lib/date/build';

import {
    swap,
    setTransportTypeFromUser,
    submitForm,
    setWhen,
} from '../../actions/searchForm';

import Icon from '../Icon/Icon';
import Button2 from '../Button2/Button2';
import Spinner from '../Spinner/Spinner';
import DateButtons from '../DateButtons';
import Transports from './Transports.mobile';
import ClearButton from '../basic/ClearButton';
import SearchFormCalendar from '../SearchFormCalendar/SearchFormCalendar';
import SearchFormDateButton from '../SearchFormDateButton/SearchFormDateButton';

import searchFormKeyset from '../../i18n/search-form';
import isAllDaysSearch from '../../lib/search/isAllDaysSearch';

const b = B('SearchForm');
const SPECIAL_BUTTONS = [DateSpecialValue.today, DateSpecialValue.tomorrow];

const mapStateToProps = ({language, flags, searchForm, currencies, page}) => ({
    language,
    flags,
    searchForm,
    currencies,
    page,
});

class SearchForm extends PureComponent {
    static defaultProps = {
        className: '',
        onOpenDatepicker: noop,
        onCloseDatepicker: noop,
        onSuggestSubmit: noop,
        onBlur: noop,
        hideSelectorOfTransportType: false,
    };

    constructor(props) {
        super();

        const {showOnlyOpenedForm} = props;

        this.state = {
            opened: showOnlyOpenedForm,
            afterEvent: undefined,
            datepickerIsOpened: false,
            inputFromStyle: {},
            inputToStyle: {},
        };
    }

    componentWillReceiveProps(newProps) {
        if (newProps.showOnlyOpenedForm) {
            if (!this.state.opened) {
                this._toggleView(false);

                this.setState({
                    afterEvent: 'opened',
                });
            }
        } else if (newProps.opened !== this.props.opened) {
            this._toggleView(!newProps.opened);

            if (newProps.opened) {
                this.setState({
                    afterEvent: 'opened',
                });
            } else {
                this.setState({
                    afterEvent: 'minimized',
                });
            }
        }
    }

    onChangeDate = (e, {value}) => {
        const {
            searchForm: {when},
        } = this.props;

        if (value.date !== when.date) {
            reachGoal('search_form_change_parameters');
        }

        reachGoal('calendar_toolbar_button_click', {date: value.date});
        this.props.onChange('when', value);
    };

    onChangeFrom = e => {
        reachGoal('search_form_input_change_value');
        this.onChange('from', e);
    };

    onFocusFrom = e => {
        const {
            searchForm: {userInput},
        } = this.props;

        reachGoal('search_form_field_from_click', {
            currentValue: userInput.from.title,
        });

        this.props.onFocus('from', e);
        document.addEventListener('keydown', this.onKeyPress);
    };

    onChangeTo = e => {
        reachGoal('search_form_input_change_value');
        this.onChange('to', e);
    };

    onFocusTo = e => {
        const {
            searchForm: {userInput},
        } = this.props;

        reachGoal('search_form_field_to_click', {
            currentValue: userInput.to.title,
        });

        this.props.onFocus('to', e);
        document.addEventListener('keydown', this.onKeyPress);
    };

    onBlur = e => {
        this.props.onBlur(e);
        document.removeEventListener('keydown', this.onKeyPress);
    };

    onKeyPress = e => {
        const pressedKey = e.which || e.keyCode;

        if (pressedKey === KeyCode.enter) {
            e.preventDefault();
            e.target.blur();
            this.props.onSuggestSubmit();
        }
    };

    onTransportSelect = type => e => {
        reachGoal('search_form_change_parameters');
        reachGoal('search_transport_click', {transportType: type});
        this.props.dispatch(
            setTransportTypeFromUser(e.currentTarget.dataset.code),
        );
        e.preventDefault();
    };

    onClearClick = () => {
        const {onChange, suggestField} = this.props;
        const input = this[`_${suggestField}`];

        reachGoal('search_form_clear_button_click');
        onChange(suggestField, {
            title: '',
            key: '',
        });

        if (input) {
            input.focus();
            input.setSelectionRange(0, 0);
        }
    };

    onOpen = () => {
        this.props.onOpen();
    };

    onSubmit = e => {
        reachGoal('point_point_search_button_click');
        this.props.onClose(true);

        if (e.defaultPrevented) {
            return;
        }

        e.preventDefault();

        this.props.dispatch(submitForm());
    };

    onChange = (field, e) => {
        const data = {
            title: e.target.value,
            key: '',
        };

        this.props.onChange(field, data);
    };

    onDatepickerClick = () => {
        const {onOpenDatepicker} = this.props;

        reachGoal('search_form_open_datepicker_click');
        this.setState({datepickerIsOpened: true});
        onOpenDatepicker();
    };

    onSpecialDateClick = special => {
        const {
            searchForm: {time},
            dispatch,
            language,
        } = this.props;

        reachGoal('search_form_change_parameters');

        if (special) {
            const when = buildSpecial(special, {language, time});

            reachGoal(`search_form_${special}_click`);
            dispatch(setWhen(when));
        }
    };

    onDatepickerClose = () => {
        const {onCloseDatepicker} = this.props;

        this.setState({datepickerIsOpened: false});
        onCloseDatepicker();
    };

    onSwitch = () => {
        reachGoal('search_form_change_parameters');
        reachGoal('point_point_swap_button_click');
        this.props.dispatch(swap());
    };

    onContainerClick = () => {
        const {opened} = this.state;

        if (!opened) {
            reachGoal('search_form_expand_search_form_click');
        }
    };

    fromSpanRef = el => {
        this._fromSpan = el;
    };

    toSpanRef = el => {
        this._toSpan = el;
    };

    _toggleView(close) {
        if (close) {
            reachGoal('search_form_tap_close');
        }

        const inputFromStyle = {
            width: this._fromSpan.offsetWidth,
            transform: `translate(${this._fromSpan.offsetLeft}px, ${this._fromSpan.offsetTop}px)`,
            transition: !close && 'none',
        };

        const inputToStyle = {
            width: this._toSpan.offsetWidth,
            transform: `translate(${this._toSpan.offsetLeft}px, ${this._toSpan.offsetTop}px)`,
            transition: !close && 'none',
        };

        const nextState = close
            ? {
                  opened: false,
              }
            : {
                  opened: true,
                  inputFromStyle: {},
                  inputToStyle: {},
              };

        this.setState(
            {
                inputFromStyle,
                inputToStyle,
            },
            () => setImmediate(() => this.setState(nextState)),
        );
    }

    getSearchContext() {
        const {searchForm} = this.props;
        const {from, to} = searchForm;
        const isAllDays = isAllDaysSearch(searchForm);

        return (
            <div className={b('layer', {order: 1})} onClick={this.onOpen}>
                <header className={b('top-line', {collapsed: true})}>
                    <span className={b('title', {canWrap: !isAllDays})}>
                        <span
                            className={b('point', {empty: !from.title})}
                            ref={this.fromSpanRef}
                        >
                            {from.title || searchFormKeyset('label-from')}
                        </span>

                        {`${CHAR_NBSP}${CHAR_EM_DASH}${CHAR_NBSP}`}

                        <span
                            className={b('point', {empty: !to.title})}
                            ref={this.toSpanRef}
                        >
                            {to.title || searchFormKeyset('label-to')}
                        </span>
                    </span>
                </header>

                {isAllDays && (
                    <div className={b('bottomLine')}>
                        <span
                            className={b('bottomLineItem', {time: true})}
                            data-nosnippet
                        >
                            {searchForm.when.text}
                        </span>
                    </div>
                )}

                <Icon
                    glyph={IconGlyph.search}
                    className={`${b('search-icon')} ${b('animated', {
                        center: true,
                    })}`}
                />
            </div>
        );
    }

    renderDateButton() {
        const {searchForm} = this.props;
        const {opened} = this.state;
        const isAllDays = isAllDaysSearch(searchForm);

        if (isAllDays || opened) {
            return null;
        }

        return <SearchFormDateButton />;
    }

    render() {
        const {
            context,
            language,
            minimized,
            searchForm,
            suggestField,
            inputsContainerRef,
            flags,
            className,
            showOnlyOpenedForm,
            hideSelectorOfTransportType,
        } = this.props;

        const {userInput, when, time, validation} = searchForm;
        const {from, to} = userInput;

        const {
            afterEvent,
            opened,
            inputFromStyle,
            inputToStyle,
            datepickerIsOpened,
        } = this.state;

        return (
            <form
                className={b(
                    {
                        opened,
                        minimized,
                        afterEvent,
                        showOnlyOpenedForm,
                        hideSelectorOfTransportType,
                    },
                    className,
                )}
                method="get"
                onSubmit={this.onSubmit}
                autoComplete="off"
            >
                <SearchFormCalendar
                    time={time}
                    value={when}
                    fixed={false}
                    context={context}
                    opened={datepickerIsOpened}
                    language={language}
                    onClose={this.onDatepickerClose}
                    onChange={this.onChangeDate}
                    flags={flags}
                />
                {!suggestField && (
                    <div className={b('buttons')}>
                        <div className={b('options')}>
                            <DateButtons
                                whenSpecial={when.special}
                                whenDate={when.date}
                                onSpecialDateClick={this.onSpecialDateClick}
                                onDatepickerClick={this.onDatepickerClick}
                                specialButtons={SPECIAL_BUTTONS}
                            />
                            {!hideSelectorOfTransportType && (
                                <Transports
                                    className={b('transports')}
                                    searchForm={searchForm}
                                    onSelect={this.onTransportSelect}
                                    language={language}
                                />
                            )}
                        </div>

                        <Button2
                            className={b('searchButton')}
                            theme="action"
                            type="submit"
                        >
                            {(validation && (
                                <Spinner
                                    className={b('spinner')}
                                    size="small"
                                    color="gray"
                                />
                            )) ||
                                searchFormKeyset('find')}
                        </Button2>
                    </div>
                )}

                <div className={b('container')} onClick={this.onContainerClick}>
                    {this.getSearchContext()}
                    <div className={b('layer', {order: 2})}>
                        <span
                            className={`${b('middle-line')} ${b('animated', {
                                center: true,
                            })}`}
                        />
                        <Icon
                            glyph={IconGlyph.switch}
                            onClick={this.onSwitch}
                            className={`${b('switch-icon')} ${b('animated', {
                                center: true,
                            })}`}
                        />
                    </div>

                    <div
                        className={b('layer', {order: 3})}
                        ref={inputsContainerRef}
                    >
                        <div className={b('top-line')}>
                            <input
                                className={b('input', {
                                    type: 'from',
                                    suggest:
                                        !showOnlyOpenedForm &&
                                        suggestField === 'from',
                                })}
                                ref={refCallback(this, '_from')}
                                name="fromName"
                                value={from.title}
                                onChange={this.onChangeFrom}
                                onFocus={this.onFocusFrom}
                                onBlur={this.onBlur}
                                style={inputFromStyle}
                                placeholder={searchFormKeyset('label-from')}
                                autoComplete="off"
                            />

                            <input
                                className={b('input', {
                                    type: 'to',
                                    suggest:
                                        !showOnlyOpenedForm &&
                                        suggestField === 'to',
                                })}
                                ref={refCallback(this, '_to')}
                                name="toName"
                                value={to.title}
                                onChange={this.onChangeTo}
                                onFocus={this.onFocusTo}
                                onBlur={this.onBlur}
                                style={inputToStyle}
                                placeholder={searchFormKeyset('label-to')}
                                autoComplete="off"
                            />

                            <input
                                type="hidden"
                                name="fromId"
                                value={from.key}
                            />

                            <input type="hidden" name="toId" value={to.key} />

                            <input
                                type="hidden"
                                name="when"
                                value={when.date || when.text}
                            />

                            <ClearButton
                                className={b('clearButton')}
                                onClick={this.onClearClick}
                                visible={Boolean(
                                    suggestField &&
                                        userInput[suggestField].title.length >
                                            0,
                                )}
                            />
                        </div>
                    </div>
                </div>

                {this.renderDateButton()}
            </form>
        );
    }
}

export default connect(mapStateToProps)(SearchForm);
