import React, {useCallback, useState} from 'react';
import {batch} from 'react-redux';
import B from 'bem-cn-lite';
import moment, {Moment} from 'moment';

import {HUMAN_WEEKDAY_SHORT, SHORT_DATE} from '../../lib/date/formats';

import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
import IconGlyph from '../../interfaces/components/IconGlyph';

import {buildSpecial} from '../../lib/date/build';
import {reachGoal} from '../../lib/yaMetrika';
import {getCurrentDay, getNextDay} from '../../lib/date/utils';

import {setWhen, submitForm} from '../../actions/searchForm';

import useSelector from '../useSelector';
import useDispatch from '../useDispatch';
import {useFlags} from '../../hooks/useFlags';

import Icon from '../Icon/Icon';
import Button2 from '../Button2/Button2';
import SearchFormCalendar from '../SearchFormCalendar/SearchFormCalendar';

import datePickerKeyset from '../../i18n/segment-date-picker';
import searchFormKeyset from '../../i18n/search-form';

const b = B('SearchFormDateButtons');

const SearchFormDateButtons: React.FC = () => {
    const [isCalendarVisible, setCalendarVisible] = useState(false);

    const dispatch = useDispatch();

    const language = useSelector(state => state.language);
    const flags = useFlags();
    const {time, when} = useSelector(state => state.searchForm);
    const {context} = useSelector(state => state.search);
    const {from} = context;
    const timezone = from.timezone || moment.tz.guess();
    const today = getCurrentDay(timezone);
    const tomorrow = getNextDay(timezone);

    const handleClick = useCallback(
        special => {
            reachGoal(`search_form_${special}_button_click`);

            const newWhen = buildSpecial(special, {language, time});

            batch(() => {
                dispatch(setWhen(newWhen));
                dispatch(submitForm());
            });
        },
        [dispatch, language, time],
    );

    const handleTodayClick = useCallback(
        () => handleClick(DateSpecialValue.today),
        [handleClick],
    );

    const handleTomorrowClick = useCallback(
        () => handleClick(DateSpecialValue.tomorrow),
        [handleClick],
    );

    const handleSelectDateClick = useCallback(() => {
        reachGoal('search_form_open_datepicker_click');
        setCalendarVisible(true);
    }, [setCalendarVisible]);

    const handleCalendarDateChange = useCallback(
        (e, {value}) => {
            reachGoal('search_form_select_date_button_click');
            batch(() => {
                dispatch(setWhen(value));
                dispatch(submitForm());
            });
        },
        [dispatch],
    );

    const handleCalendarClose = useCallback(() => {
        setCalendarVisible(false);
    }, [setCalendarVisible]);

    return (
        <div className={b()}>
            <Button2
                theme="action"
                onClick={handleSelectDateClick}
                className={b('calendarButton')}
                iconRight={
                    <Icon
                        glyph={IconGlyph.calendarSmall}
                        className={b('calendarIcon')}
                    />
                }
            >
                {datePickerKeyset('button-text')}
            </Button2>
            <div onClick={handleTodayClick} className={b('dayButton')}>
                <div className={b('dayButtonTitle')}>
                    {searchFormKeyset('today')}
                </div>
                <div className={b('dayButtonDescription')}>
                    {formatDate(today)}
                </div>
            </div>
            <div className={b('delimiter')} />
            <div onClick={handleTomorrowClick} className={b('dayButton')}>
                <div className={b('dayButtonTitle')}>
                    {searchFormKeyset('tomorrow')}
                </div>
                <div className={b('dayButtonDescription')}>
                    {formatDate(tomorrow)}
                </div>
            </div>

            <SearchFormCalendar
                time={time}
                value={when}
                fixed={false}
                context={context}
                opened={isCalendarVisible}
                language={language}
                onClose={handleCalendarClose}
                onChange={handleCalendarDateChange}
                flags={flags}
                showToolbar={false}
            />
        </div>
    );
};

function formatDate(date: Moment): string {
    return `${date.format(HUMAN_WEEKDAY_SHORT)}. ${date.format(SHORT_DATE)}`;
}

export default SearchFormDateButtons;
