import {React, PureComponent, B, bindMethods} from '../base';

import insertMarkupIntoKey from '../../i18n/_utils/insertMarkupIntoKey';

import Popup from '../Popup/Popup';
import Button from '../Button/Button';
import SearchFormHint from '../SearchFormHint/SearchFormHint';
import DateHints from '../DateHints/DateHints';

import keyset from '../../i18n/search-hints';

const b = B('SearchFormHints');

const MAX_VISIBLE_STATION_HINTS = 3;
const MAX_VISIBLE_DIRECTIONS_HINTS = 2;
const MAX_STATION_HINTS = 13;

export default class SearchFormHints extends PureComponent {
    constructor(props) {
        super(props);

        this.state = {
            popupVisible: false,
        };

        bindMethods(this, [
            'makeMoreRef',
            'onClickOutside',
            'onPopupTogglerClick',
        ]);
    }

    onPopupTogglerClick() {
        this.setState({
            popupVisible: !this.state.popupVisible,
        });
    }

    onClickOutside(e) {
        const element = this._more;

        if (element !== e.target && !element.contains(e.target)) {
            this.setState({
                popupVisible: false,
            });
        }
    }

    makeMoreRef(node) {
        this._more = node;
    }

    getHintKey(hint) {
        return `${hint.value.from.key}_${hint.value.to.key}`;
    }

    renderHistoryHints() {
        const {searchFormHints, dispatch} = this.props;

        const {sampleSearchContext, searchHistory} = searchFormHints;

        const popupHistoryStationHints = searchHistory.slice(
            MAX_VISIBLE_STATION_HINTS,
            MAX_STATION_HINTS,
        );
        const visibleHistoryStationHints = searchHistory.slice(
            0,
            MAX_VISIBLE_STATION_HINTS,
        );
        const hasHistoryHints = searchHistory.length > 0;

        return hasHistoryHints ? (
            <div className={b('historyStationHints')}>
                {visibleHistoryStationHints.map(hint => (
                    <SearchFormHint
                        key={this.getHintKey(hint)}
                        value={hint.value}
                        favorite={hint.favorite}
                        canFavorite
                        dispatch={dispatch}
                    />
                ))}

                {popupHistoryStationHints.length > 0 && (
                    <span className={b('more')} ref={this.makeMoreRef}>
                        <Button
                            className={b('dots')}
                            onClick={this.onPopupTogglerClick}
                        >
                            &#9679; &#9679; &#9679;
                        </Button>

                        <Popup
                            visible={this.state.popupVisible}
                            showArrow={false}
                            onClickOutside={this.onClickOutside}
                        >
                            {popupHistoryStationHints.map(hint => (
                                <SearchFormHint
                                    key={this.getHintKey(hint)}
                                    value={hint.value}
                                    favorite={hint.favorite}
                                    canFavorite
                                    dispatch={dispatch}
                                />
                            ))}
                        </Popup>
                    </span>
                )}
            </div>
        ) : sampleSearchContext ? (
            this.renderSampleSearchContext(sampleSearchContext)
        ) : null;
    }

    renderSampleSearchContext(sampleSearchContext) {
        const {dispatch} = this.props;
        const hint = {
            value: {
                from: sampleSearchContext.point_from,
                to: sampleSearchContext.point_to,
            },
        };

        return (
            <div className={b('stationSampleHint')}>
                {insertMarkupIntoKey(keyset, 'for-example', {
                    hint: (
                        <SearchFormHint
                            key={this.getHintKey(hint)}
                            dispatch={dispatch}
                            value={hint.value}
                            canFavorite={false}
                        />
                    ),
                })}
            </div>
        );
    }

    renderPopularDirectionsHints() {
        const {
            dispatch,
            transportType,
            searchFormHints: {popularDirections},
        } = this.props;
        const visiblePopularDirectionsHints = popularDirections.slice(
            0,
            MAX_VISIBLE_DIRECTIONS_HINTS,
        );

        return (
            <div className={b('historyStationHints')}>
                {`${keyset(`popular-directions-${transportType}`)}: `}
                {visiblePopularDirectionsHints.map(hint => (
                    <SearchFormHint
                        key={this.getHintKey(hint)}
                        value={hint.value}
                        canFavorite={false}
                        dispatch={dispatch}
                    />
                ))}
            </div>
        );
    }

    render() {
        const {
            time,
            searchFormHints: {popularDirections, searchHistory},
            language,
            dispatch,
        } = this.props;

        const hasPopularDirections =
            popularDirections && popularDirections.length > 0;

        return (
            <div className={b()}>
                {hasPopularDirections
                    ? this.renderPopularDirectionsHints()
                    : searchHistory && this.renderHistoryHints()}

                <DateHints
                    className={b('dateHints')}
                    time={time}
                    language={language}
                    dispatch={dispatch}
                />
            </div>
        );
    }
}
