import React from 'react';
import B from 'bem-cn-lite';

import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import {TransportType} from '../../lib/transportType';
import ITime from '../../interfaces/state/searchForm/ITime';
import ISearchSort from '../../interfaces/state/search/ISearchSort';
import IStateCurrencies from '../../interfaces/state/IStateCurrencies';
import SearchSegment from '../../interfaces/state/search/SearchSegment';
import ISearchContext from '../../interfaces/state/search/ISearchContext';
import {IWithClassName} from '../../interfaces/components/IWithClassName';
import ISearchFiltering from '../../interfaces/state/search/ISearchFiltering';
import PrimaryPosition from '../../interfaces/lib/dimensions/PrimaryPosition';
import IStateSearchCalendar from '../../interfaces/state/IStateSearchCalendar';
import SecondaryPosition from '../../interfaces/lib/dimensions/SecondaryPosition';

import isAllDaysSearch from '../../lib/search/isAllDaysSearch';
import {anyFiltersAvailable} from '../../lib/filters/filtering';
import arePricesPossibleForContext from '../../lib/search/arePricesPossibleForContext';

import useDispatch from '../useDispatch';
import setSearchUrl from '../../actions/search/setSearchUrl';

import LocalTime from '../LocalTime/LocalTime';
import DateHintsLinks from '../DateHintsLinks';
import Sorting from './components/Sorting/Sorting';
import PrintButton from '../PrintButton/PrintButton';
import SearchTitle from '../SearchTitle/SearchTitle';
import ServiceDescription from '../ServiceDescription';
import LinkBackward from '../LinkBackward/LinkBackward';
import FiltersToolbar from '../FiltersToolbar/FiltersToolbar';
import CurrencySelect from '../CurrencySelect/CurrencySelect';
import SearchCalendar from '../SearchCalendar/SearchCalendar';
import TransportSelectorLinks from './components/TransportSelector/TransportSelector';

const b = B('SearchHeader');

interface ISearchHeaderProps extends IWithClassName {
    time: ITime;
    sort: ISearchSort;
    plans: object;
    context: ISearchContext;
    filtering: ISearchFiltering;
    segments: SearchSegment[];
    transportTypes: TransportType[];
    currencies: IStateCurrencies;
    language: Lang;
    tld: Tld;
    searchCalendar: IStateSearchCalendar;
    isSuburbanSearchResult: boolean;

    queryingPrices?: boolean;
    hasDescription?: boolean;
}

const CURRENCY_POPUP_POSITIONS = [
    [PrimaryPosition.bottom, SecondaryPosition.right],
    [PrimaryPosition.bottom, SecondaryPosition.left],
    [PrimaryPosition.top, SecondaryPosition.right],
    [PrimaryPosition.top, SecondaryPosition.left],
];

const SearchHeader: React.FC<ISearchHeaderProps> = ({
    className,
    time,
    sort,
    plans,
    context,
    filtering,
    segments,
    transportTypes,
    currencies,
    language,
    tld,
    searchCalendar,
    isSuburbanSearchResult,

    queryingPrices,
    hasDescription,
}) => {
    const dispatch = useDispatch();
    const {availableCurrencies = []} = currencies;

    return (
        <div className={b({}, className)}>
            <TransportSelectorLinks
                className={b('transportSelector')}
                sort={sort}
                context={context}
                filtering={filtering}
                currencies={currencies}
                tld={tld}
                language={language}
            />

            <SearchTitle className={b('title')} />

            {hasDescription && <ServiceDescription />}

            {isAllDaysSearch(context) && (
                <SearchCalendar
                    {...searchCalendar}
                    sort={sort}
                    context={context}
                    dispatch={dispatch}
                    segments={segments}
                    language={language}
                    filtering={filtering}
                    currencies={currencies}
                    isSuburbanSearchResult={isSuburbanSearchResult}
                />
            )}

            <div className={b('toolbar')}>
                <div className={b('sortingAndFilters')}>
                    <Sorting sort={sort} />

                    {(anyFiltersAvailable(filtering) || queryingPrices) && (
                        <FiltersToolbar
                            sort={sort}
                            segments={segments}
                            filtering={filtering}
                            transportTypes={transportTypes}
                            queryingPrices={queryingPrices}
                            language={language}
                            isRedesigned
                        />
                    )}
                </div>

                <div className={b('printAndCurrency')}>
                    <LocalTime className={b('localTime')} />

                    <PrintButton className={b('print')} compact />

                    {arePricesPossibleForContext(context) &&
                        availableCurrencies.length > 1 && (
                            <CurrencySelect
                                {...currencies}
                                trackFunction={setSearchUrl}
                                dispatch={dispatch}
                                language={language}
                                popupPositions={CURRENCY_POPUP_POSITIONS}
                            />
                        )}
                </div>
            </div>

            <div className={b('links')}>
                <DateHintsLinks
                    context={context}
                    sort={sort}
                    time={time}
                    plans={plans}
                    filtering={filtering}
                    currencies={currencies}
                    isRedesigned
                />

                <LinkBackward
                    sort={sort}
                    context={context}
                    filtering={filtering}
                    currencies={currencies}
                    isRedesigned
                />
            </div>
        </div>
    );
};

export default SearchHeader;
