import React, {SyntheticEvent, useCallback, useMemo} from 'react';
import upperFirst from 'lodash/upperFirst';
import B from 'bem-cn-lite';

import {SEARCHABLE_TRANSPORT_TYPES} from '../../../../lib/transportType';

import Tld from '../../../../interfaces/Tld';
import Lang from '../../../../interfaces/Lang';
import ISearchSort from '../../../../interfaces/state/search/ISearchSort';
import IStateCurrencies from '../../../../interfaces/state/IStateCurrencies';
import ISearchContext from '../../../../interfaces/state/search/ISearchContext';
import {IWithClassName} from '../../../../interfaces/components/IWithClassName';
import ISearchFiltering from '../../../../interfaces/state/search/ISearchFiltering';

import {reachGoal} from '../../../../lib/yaMetrika';
import searchUrl from '../../../../lib/url/searchUrl';
// eslint-disable-next-line no-duplicate-imports
import {FilterTransportType} from '../../../../lib/transportType';

import Link from '../../../Link';

import transportTypesKeyset from '../../../../i18n/transport-types';
import transportSelectorKeyset from '../../../../i18n/transport-selector';

interface ITransportSelectorProps extends IWithClassName {
    context: ISearchContext;
    sort: ISearchSort;
    filtering: ISearchFiltering;
    currencies: IStateCurrencies;
    language: Lang;
    tld: Tld;
}

const b = B('HeaderTransportSelector');

const TransportSelector: React.FC<ITransportSelectorProps> = ({
    sort,
    context,
    className,
    filtering,
    currencies,
    tld,
    language,
}) => {
    const {transportType: currentValue, userInput} = context;
    const {from, to} = userInput || context;

    const handleClick = useCallback((e: SyntheticEvent<HTMLElement>) => {
        reachGoal('TRANSPORT_SELECTOR_LINK_CLICK', {
            type: e.currentTarget.dataset.code as FilterTransportType,
        });
    }, []);

    const transportTypeLinks = useMemo(
        () =>
            SEARCHABLE_TRANSPORT_TYPES.map(code => ({
                value: code,
                label: upperFirst(
                    transportTypesKeyset(
                        code === 'all' ? 'type-all-long' : `type-${code}`,
                    ),
                ),
                title: transportSelectorKeyset(`title-${code}`, {
                    from: from.title,
                    to: to.title,
                }),
                url: searchUrl(
                    {
                        context: {
                            ...context,
                            transportType: code,
                        },
                        sort,
                        filtering,
                        currencies,
                    },
                    tld,
                    language,
                ),
            })),
        [context, currencies, filtering, from, language, sort, tld, to],
    );

    return (
        <div className={b({}, className)}>
            {transportTypeLinks.map((item, index) =>
                item.value === currentValue ? (
                    <span key={index} className={b('item', {selected: true})}>
                        {item.label}
                    </span>
                ) : (
                    <Link
                        key={index}
                        href={item.url}
                        colors="textSecondary"
                        title={item.title}
                        data-code={item.value}
                        onClick={handleClick}
                        className={b('item', {link: true})}
                    >
                        {item.label}
                    </Link>
                ),
            )}
        </div>
    );
};

export default TransportSelector;
