import React, {ReactElement} from 'react';

import SearchSegment from '../../interfaces/state/search/SearchSegment';
import ISearchContext from '../../interfaces/state/search/ISearchContext';
import {IWithClassName} from '../../interfaces/components/IWithClassName';

import {reachGoal} from '../../lib/yaMetrika';
import isTransfer from '../../lib/segments/isTransfer';
import {trainIncompleteScheduleDisclaimerIsAvailable} from '../../lib/disclaimers/disclaimersAvailability';

import BaseDisclaimer from '../Disclaimers/BaseDisclaimer';

import disclaimersKeyset from '../../i18n/disclaimers';

interface IDisclaimer {
    type: string;
    needShow: (context: ISearchContext, segments: SearchSegment[]) => boolean;
    getContent: (
        context: ISearchContext,
        segments: SearchSegment[],
    ) => ReactElement;
}

const getDisclaimers = (redesigned?: boolean): IDisclaimer[] => [
    {
        type: 'generalization',
        needShow(context) {
            return Boolean(context.isChanged);
        },
        getContent(context) {
            return (
                <BaseDisclaimer
                    attention={!redesigned}
                    isRedesigned={redesigned}
                    key="generalization"
                    content={disclaimersKeyset('generalization', {
                        originFrom:
                            (context.original || {pointFrom: {}}).pointFrom
                                .titleWithType || '',
                        originTo:
                            (context.original || {pointTo: {}}).pointTo
                                .titleWithType || '',
                        searchFrom: context.from.titleWithType || '',
                        searchTo: context.to.titleWithType || '',
                    })}
                />
            );
        },
    },
    {
        type: 'train-incomplete-schedule',
        needShow(context, segments) {
            return trainIncompleteScheduleDisclaimerIsAvailable(
                context,
                segments,
            );
        },
        getContent() {
            return (
                <BaseDisclaimer
                    attention={!redesigned}
                    isRedesigned={redesigned}
                    key="train-incomplete-schedule"
                    content={disclaimersKeyset('train-incomplete-schedule')}
                />
            );
        },
    },
    {
        type: 'no-direct-routes',
        needShow(context, segments) {
            return segments.every(segment => isTransfer(segment));
        },
        getContent() {
            return (
                <BaseDisclaimer
                    attention={!redesigned}
                    isRedesigned={redesigned}
                    key="no-direct-routes"
                    content={disclaimersKeyset('no-direct-routes')}
                />
            );
        },
    },
];

interface ISearchHeaderDisclaimersProps extends IWithClassName {
    context: ISearchContext;
    segments: SearchSegment[];

    redesigned?: boolean;
}

const SearchHeaderDisclaimers: React.FC<ISearchHeaderDisclaimersProps> = ({
    context,
    segments,

    className,
    redesigned,
}) => {
    const disclaimers = getDisclaimers(redesigned).reduce(
        (elements: ReactElement[], disclaimer: IDisclaimer) => {
            if (disclaimer.needShow(context, segments)) {
                elements.push(disclaimer.getContent(context, segments));

                reachGoal('disclaimer_shown', {
                    context: {
                        from: context.userInput.from,
                        to: context.userInput.to,
                        transportType: context.transportType,
                        date: context.when.date,
                    },
                    disclaimerType: disclaimer.type,
                });
            }

            return elements;
        },
        [],
    );

    if (!disclaimers.length) {
        return null;
    }

    return <div className={className}>{disclaimers}</div>;
};

export default SearchHeaderDisclaimers;
