import React, {ReactElement, memo} from 'react';
import moment from 'moment';
import B from 'bem-cn-lite';
import {upperFirst} from 'lodash';

import {TIME, HUMAN} from '../../lib/date/formats';
import {SEARCH_INFORMER_UTM_MEDIUM} from '../../lib/url/constants';

import {TransportType, FilterTransportType} from '../../lib/transportType';
import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
import ISearchInformer from '../../interfaces/components/informer/ISearchInformer';
import IconGlyphInformers from '../../interfaces/components/IconGlyphInformers';

import needShowDate from './needShowDate';
import {isCity} from '../../lib/point/pointType';
import getSearchUrl from '../../lib/url/getSearchUrl';
import applyInformerUtm from '../../lib/url/applyInformerUtm';
import getSubtype from '../../lib/segments/transportFeatures/getSubtype';
import getStaticTransportTypeName from '../../lib/getStaticTransportTypeName';
import getThreadUrlForSegment from '../../lib/segments/getThreadUrlForSegment';

import Link from '../Link';
import Icon from '../Icon/Icon';
import TransportIcon from '../TransportIcon/TransportIcon';
import SearchInformerH1 from './SearchInformerH1/SearchInformerH1';
import InformerDirectionTitle from '../InformerDirectionTitle/InformerDirectionTitle';

import informerKeyset from '../../i18n/informers';
import transportTypesKeyset from '../../i18n/transport-types';

const b = B('SearchInformerDesktop');

export default memo(SearchInformerDesktop);

function SearchInformerDesktop({
    segments,
    tld,
    language,
    transportType,
    pointFrom,
    pointTo,
    dateMomentOnStationFrom,

    color,
    size,
}: ISearchInformer): ReactElement {
    const emptyRows: null[] =
        size && segments.length < size
            ? Array(size - segments.length).fill(null)
            : [];

    const searchUrlParams = {
        transportType,
        tld,
        language,
        fromPoint: pointFrom.key,
        toPoint: pointTo.key,
        fromSlug: pointFrom.slug,
        toSlug: pointTo.slug,
    };

    return (
        <div className={b({color})}>
            <div className={b('header')}>
                <Link
                    className={b('mainPageLink')}
                    href={applyInformerUtm('/', SEARCH_INFORMER_UTM_MEDIUM)}
                    target="_blank"
                >
                    <Icon
                        className={b('serviceIcon')}
                        glyph={IconGlyphInformers.serviceDescription}
                    />

                    {informerKeyset('main-page-link')}
                </Link>

                <SearchInformerH1
                    pointFrom={pointFrom}
                    pointTo={pointTo}
                    transportType={transportType}
                    tld={tld}
                    language={language}
                    dateMomentOnStationFrom={dateMomentOnStationFrom}
                />
            </div>

            <table className={b('table')}>
                <tbody>
                    {segments.map((segment, index) => {
                        const {
                            stationFrom,
                            stationTo,
                            title: segmentTitle,
                            thread,
                            transport,
                            departureLocalDt,
                            arrivalLocalDt,
                        } = segment;

                        const threadUrl = getThreadUrlForSegment({
                            segment,
                            tld,
                            language,
                            isToCitySearchContext: isCity(pointTo),
                        });
                        const threadUrlWithUtm = threadUrl
                            ? applyInformerUtm(
                                  threadUrl,
                                  SEARCH_INFORMER_UTM_MEDIUM,
                              )
                            : undefined;

                        const subtype = getSubtype(
                            transport,
                            thread,
                            transportType === FilterTransportType.suburban,
                        );

                        const departureTime =
                            moment.parseZone(departureLocalDt);
                        const arrivalTime = moment.parseZone(arrivalLocalDt);

                        const prevSegment = segments[index - 1];

                        const {
                            arrival: showArrivalDate,
                            departure: showDepatrureDate,
                        } = needShowDate(segment, prevSegment);
                        const needShowArrivalDate =
                            showArrivalDate || index === 0;
                        const needShowDepartureDate =
                            showDepatrureDate || index === 0;

                        const needShowTransportType =
                            transport.code !== TransportType.water || !subtype;

                        return (
                            <tr className={b('threadRow')} key={index}>
                                <td className={b('iconColumn')}>
                                    <TransportIcon
                                        className={b('icon')}
                                        transportType={transport.code}
                                        isExpress={thread?.isExpress}
                                        isAeroExpress={thread?.isAeroExpress}
                                        colored
                                    />
                                </td>

                                <td className={b('directionColumn')}>
                                    <div className={b('threadName')}>
                                        <InformerDirectionTitle
                                            title={segmentTitle}
                                            href={threadUrlWithUtm}
                                        />

                                        <span className={b('threadNumber')}>
                                            {thread?.number}
                                        </span>
                                    </div>

                                    <div className={b('transportType')}>
                                        {needShowTransportType && (
                                            <span
                                                className={b(
                                                    'transportTypeDefault',
                                                )}
                                            >
                                                {upperFirst(
                                                    !transport.code
                                                        ? transportTypesKeyset(
                                                              'type-all-long',
                                                          )
                                                        : getStaticTransportTypeName(
                                                              transport.code,
                                                          ),
                                                )}
                                            </span>
                                        )}

                                        {thread?.deluxeTrain && !subtype && (
                                            <span className={b('deluxeTrain')}>
                                                {thread?.deluxeTrain.shortTitle}
                                            </span>
                                        )}

                                        {subtype && (
                                            <span
                                                className={b('subtype')}
                                                style={subtype.props.style}
                                                dangerouslySetInnerHTML={{
                                                    __html: subtype.content,
                                                }}
                                            />
                                        )}
                                    </div>
                                </td>

                                <td className={b('stationFromColumn')}>
                                    <span className={b('stationFromDate')}>
                                        <span className={b('stationFromTime')}>
                                            {departureTime.format(TIME)}
                                        </span>
                                        {needShowDepartureDate && (
                                            <span className={b('segmentDate')}>
                                                {`${departureTime.format(
                                                    HUMAN,
                                                )}`}
                                            </span>
                                        )}
                                    </span>

                                    <div className={b('station')}>
                                        {stationFrom.popularTitle ||
                                            stationFrom.title}
                                    </div>
                                </td>

                                <td className={b('stationToColumn')}>
                                    <span className={b('stationToDate')}>
                                        <span className={b('stationToTime')}>
                                            {arrivalTime.format(TIME)}
                                        </span>
                                        {needShowArrivalDate && (
                                            <span className={b('segmentDate')}>
                                                {`${arrivalTime.format(HUMAN)}`}
                                            </span>
                                        )}
                                    </span>

                                    <div className={b('station')}>
                                        {stationTo.popularTitle ||
                                            stationTo.title}
                                    </div>
                                </td>
                            </tr>
                        );
                    })}

                    {emptyRows.map((e, index) => (
                        <tr
                            className={b('threadRow', {empty: true})}
                            key={index}
                        >
                            <td
                                colSpan={4}
                                className={b('noSegmentsColumn', {
                                    noSegmentsMessage:
                                        !segments.length && index === 0,
                                })}
                            >
                                {Boolean(!segments.length && index === 0) &&
                                    informerKeyset('no-segments')}
                            </td>
                        </tr>
                    ))}
                </tbody>
            </table>

            <div className={b('footer')}>
                <div className={b('fullScheduleLinks')}>
                    <span className={b('fullSchedule')}>
                        {`${informerKeyset('full-schedule')} `}
                    </span>

                    <Link
                        className={b('fullScheduleLink')}
                        href={
                            searchUrlParams
                                ? applyInformerUtm(
                                      getSearchUrl({
                                          ...searchUrlParams,
                                          when: DateSpecialValue.today,
                                      }),
                                      SEARCH_INFORMER_UTM_MEDIUM,
                                  )
                                : undefined
                        }
                        target="_blank"
                    >
                        {informerKeyset('search-today')}
                    </Link>

                    <Link
                        className={b('fullScheduleLink')}
                        href={
                            searchUrlParams
                                ? applyInformerUtm(
                                      getSearchUrl({
                                          ...searchUrlParams,
                                          when: DateSpecialValue.tomorrow,
                                      }),
                                      SEARCH_INFORMER_UTM_MEDIUM,
                                  )
                                : undefined
                        }
                        target="_blank"
                    >
                        {informerKeyset('search-tomorrow')}
                    </Link>

                    <Link
                        className={b('fullScheduleLink')}
                        href={
                            searchUrlParams
                                ? applyInformerUtm(
                                      getSearchUrl({
                                          ...searchUrlParams,
                                          when: DateSpecialValue.allDays,
                                      }),
                                      SEARCH_INFORMER_UTM_MEDIUM,
                                  )
                                : undefined
                        }
                        target="_blank"
                    >
                        {informerKeyset('search-all-days')}
                    </Link>
                </div>
            </div>
        </div>
    );
}
