import React, {ReactElement, memo} from 'react';
import moment from 'moment';
import B from 'bem-cn-lite';

import {CHAR_EM_DASH} from '../../lib/stringUtils';
import {TIME, HUMAN_SHORT} from '../../lib/date/formats';
import {SEARCH_INFORMER_UTM_MEDIUM} from '../../lib/url/constants';

import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
import ISearchInformer from '../../interfaces/components/informer/ISearchInformer';
import IconGlyphInformers from '../../interfaces/components/IconGlyphInformers';
import {FilterTransportType} from '../../lib/transportType';

import needShowDate from './needShowDate';
import {isCity} from '../../lib/point/pointType';
import getSearchUrl from '../../lib/url/getSearchUrl';
import applyInformerUtm from '../../lib/url/applyInformerUtm';
import getSubtype from '../../lib/segments/transportFeatures/getSubtype';
import getThreadUrlForSegment from '../../lib/segments/getThreadUrlForSegment';
import needCutStations from './needCutStations';

import Link from '../Link';
import Icon from '../Icon/Icon';
import TransportIcon from '../TransportIcon/TransportIcon';
import SearchInformerH1 from './SearchInformerH1/SearchInformerH1';
import InformerDirectionTitle from '../InformerDirectionTitle/InformerDirectionTitle';

import informerKeyset from '../../i18n/informers';

const b = B('SearchInformerMobile');

export default memo(SearchInformerMobile);

function SearchInformerMobile({
    segments,
    tld,
    language,
    transportType,
    pointFrom,
    pointTo,
    dateMomentOnStationFrom,

    color,
    size,
}: ISearchInformer): ReactElement {
    const emptyRows: null[] =
        size && segments.length < size
            ? Array(size - segments.length).fill(null)
            : [];

    const searchUrlParams = {
        transportType,
        tld,
        language,
        fromPoint: pointFrom.key,
        toPoint: pointTo.key,
        fromSlug: pointFrom.slug,
        toSlug: pointTo.slug,
    };

    return (
        <div className={b({color})}>
            <div className={b('header')}>
                <Link
                    className={b('mainPageLink')}
                    href={applyInformerUtm('/', SEARCH_INFORMER_UTM_MEDIUM)}
                    target="_blank"
                >
                    <Icon
                        className={b('serviceIcon')}
                        glyph={IconGlyphInformers.serviceDescription}
                    />

                    {informerKeyset('main-page-link')}
                </Link>

                <SearchInformerH1
                    pointFrom={pointFrom}
                    pointTo={pointTo}
                    transportType={transportType}
                    tld={tld}
                    language={language}
                    dateMomentOnStationFrom={dateMomentOnStationFrom}
                    isMobile
                />
            </div>

            <div className={b('table')}>
                {segments.map((segment, index) => {
                    const {
                        stationFrom,
                        stationTo,
                        title: segmentTitle,
                        thread,
                        transport,
                        departureLocalDt,
                        arrivalLocalDt,
                    } = segment;

                    const threadUrl = getThreadUrlForSegment({
                        segment,
                        tld,
                        language,
                        isToCitySearchContext: isCity(pointTo),
                    });
                    const threadUrlWithUtm = threadUrl
                        ? applyInformerUtm(
                              threadUrl,
                              SEARCH_INFORMER_UTM_MEDIUM,
                          )
                        : undefined;

                    const subtype = getSubtype(
                        transport,
                        thread,
                        transportType === FilterTransportType.suburban,
                    );

                    const departureTime = moment.parseZone(departureLocalDt);
                    const arrivalTime = moment.parseZone(arrivalLocalDt);

                    const prevSegment = segments[index - 1];

                    const {
                        arrival: showArrivalDate,
                        departure: showDepatrureDate,
                    } = needShowDate(segment, prevSegment);
                    const needShowArrivalDate = showArrivalDate || index === 0;
                    const needShowDepartureDate =
                        showDepatrureDate || index === 0;

                    const {needCutStationFrom, needCutStationTo} =
                        needCutStations({
                            stationFrom:
                                stationFrom.popularTitle || stationFrom.title,
                            stationTo:
                                stationTo.popularTitle || stationTo.title,
                        });

                    return (
                        <div className={b('threadRow')} key={index}>
                            <div className={b('time')}>
                                {departureTime.format(TIME)}{' '}
                                {needShowDepartureDate && (
                                    <span className={b('segmentDate')}>
                                        {departureTime.format(HUMAN_SHORT)}
                                    </span>
                                )}
                                {` ${CHAR_EM_DASH} `}
                                {arrivalTime.format(TIME)}{' '}
                                {needShowArrivalDate && (
                                    <span className={b('segmentDate')}>
                                        {arrivalTime.format(HUMAN_SHORT)}
                                    </span>
                                )}
                            </div>

                            <div className={b('stations')}>
                                <span
                                    className={b('stationFrom', {
                                        cut: needCutStationFrom,
                                    })}
                                >
                                    {`${
                                        stationFrom.popularTitle ||
                                        stationFrom.title
                                    }`}
                                </span>
                                {CHAR_EM_DASH}
                                <span
                                    className={b('stationTo', {
                                        cut: needCutStationTo,
                                    })}
                                >
                                    {`${
                                        stationTo.popularTitle ||
                                        stationTo.title
                                    }`}
                                </span>
                            </div>

                            <div className={b('threadLink')}>
                                <TransportIcon
                                    className={b('icon')}
                                    transportType={transport.code}
                                    isExpress={thread?.isExpress}
                                    isAeroExpress={thread?.isAeroExpress}
                                    colored
                                />

                                <InformerDirectionTitle
                                    title={segmentTitle}
                                    href={threadUrlWithUtm}
                                />

                                <span className={b('threadNumber')}>
                                    {thread?.number}
                                </span>

                                {thread?.deluxeTrain && !subtype && (
                                    <span className={b('deluxeTrain')}>
                                        {thread?.deluxeTrain.shortTitle}
                                    </span>
                                )}

                                {subtype && (
                                    <span
                                        className={b('subtype')}
                                        style={subtype.props.style}
                                        dangerouslySetInnerHTML={{
                                            __html: subtype.content,
                                        }}
                                    />
                                )}
                            </div>
                        </div>
                    );
                })}

                {emptyRows.map((e, index) => (
                    <div
                        className={b('threadRow', {
                            empty: true,
                            noSegmentsMessage: !segments.length && index === 0,
                        })}
                        key={index}
                    >
                        {Boolean(!segments.length && index === 0) &&
                            informerKeyset('no-segments')}
                    </div>
                ))}
            </div>

            <div className={b('footer')}>
                <div className={b('fullScheduleLinks')}>
                    <Link
                        className={b('fullScheduleLink')}
                        href={
                            searchUrlParams
                                ? applyInformerUtm(
                                      getSearchUrl({
                                          ...searchUrlParams,
                                          when: DateSpecialValue.today,
                                      }),
                                      SEARCH_INFORMER_UTM_MEDIUM,
                                  )
                                : undefined
                        }
                        target="_blank"
                    >
                        {informerKeyset('search-today')}
                    </Link>

                    <Link
                        className={b('fullScheduleLink')}
                        href={
                            searchUrlParams
                                ? applyInformerUtm(
                                      getSearchUrl({
                                          ...searchUrlParams,
                                          when: DateSpecialValue.tomorrow,
                                      }),
                                      SEARCH_INFORMER_UTM_MEDIUM,
                                  )
                                : undefined
                        }
                        target="_blank"
                    >
                        {informerKeyset('search-tomorrow')}
                    </Link>

                    <Link
                        className={b('fullScheduleLink')}
                        href={
                            searchUrlParams
                                ? applyInformerUtm(
                                      getSearchUrl({
                                          ...searchUrlParams,
                                          when: DateSpecialValue.allDays,
                                      }),
                                      SEARCH_INFORMER_UTM_MEDIUM,
                                  )
                                : undefined
                        }
                        target="_blank"
                    >
                        {informerKeyset('search-all-days')}
                    </Link>
                </div>
            </div>
        </div>
    );
}
