import React, {memo} from 'react';
import B from 'bem-cn-lite';
import {Moment} from 'moment';

import {HUMAN} from '../../../lib/date/formats';
import {CHAR_EM_DASH} from '../../../lib/stringUtils';
import {SEARCH_INFORMER_UTM_MEDIUM} from '../../../lib/url/constants';

import Tld from '../../../interfaces/Tld';
import Lang from '../../../interfaces/Lang';
import IPoint from '../../../interfaces/state/searchForm/IPoint';
import {FilterTransportType} from '../../../lib/transportType';
import DateSpecialValue from '../../../interfaces/date/DateSpecialValue';

import getSearchUrl from '../../../lib/url/getSearchUrl';
import applyInformerUtm from '../../../lib/url/applyInformerUtm';
import needCutStations from '../needCutStations';

import Link from '../../Link';
import LocalTimeDisclaimer from '../../LocalTimeDisclaimer/LocalTimeDisclaimer';

const b = B('SearchInformerH1');

interface ISearchInformerH1 {
    pointFrom: IPoint;
    pointTo: IPoint;
    transportType: FilterTransportType;
    tld: Tld;
    language: Lang;
    dateMomentOnStationFrom: Moment;

    when?: DateSpecialValue;
    className?: string;
    isMobile?: boolean;
}

export default memo(SearchInformerH1);

function SearchInformerH1({
    pointFrom,
    pointTo,
    transportType,
    tld,
    language,
    dateMomentOnStationFrom,

    isMobile,
    when,
    className,
}: ISearchInformerH1): React.ReactElement {
    const key = isMobile ? 'mobile' : 'desktop';

    const searchUrlParams = {
        transportType,
        tld,
        language,
        fromPoint: pointFrom.key,
        toPoint: pointTo.key,
        fromSlug: pointFrom.slug,
        toSlug: pointTo.slug,
        when: when || DateSpecialValue.today,
    };
    const searchUrl = applyInformerUtm(
        getSearchUrl(searchUrlParams),
        SEARCH_INFORMER_UTM_MEDIUM,
    );

    const date = dateMomentOnStationFrom.format(HUMAN);

    const fromTitle = pointFrom.popularTitle || pointFrom.title;
    const toTitle = pointTo.popularTitle || pointTo.title;

    const {needCutStationFrom, needCutStationTo} = needCutStations({
        stationFrom: fromTitle,
        stationTo: toTitle,
    });

    return (
        <span className={b({isMobile}, className)}>
            <h1 className={b('h1')}>
                <Link
                    href={searchUrl}
                    target="_blank"
                    key={key}
                    className={b('titleLink')}
                >
                    <span className={b('direction')}>
                        <span className={b('from', {cut: needCutStationFrom})}>
                            {fromTitle}
                        </span>
                        {CHAR_EM_DASH}
                        <span className={b('to', {cut: needCutStationTo})}>
                            {toTitle}
                        </span>
                    </span>
                    <span>{date}</span>
                </Link>
            </h1>

            <LocalTimeDisclaimer className={b('localTimeDisclaimer')} />
        </span>
    );
}
