import React, {memo, useMemo} from 'react';
import B from 'bem-cn-lite';

import {TransportType} from '../../lib/transportType';
import SearchErrorPageType from '../../../common/interfaces/components/searchErrorPage/SearchErrorPageType';

import useSelector from '../useSelector';
import useDispatch from '../useDispatch';
import {useFlags} from '../../hooks/useFlags';
import isResultEmpty from '../../lib/search/isResultEmpty';
import isAllDaysSearch from '../../lib/search/isAllDaysSearch';
import {shouldShowSearchDescription} from '../../lib/seo/search';
import buildArchivalSegmentsVector from '../../lib/sort/buildArchivalSegmentsVector';
import valueForSegmentBuyButtonUtmContext from './utilities/valueForSegmentBuyButtonUtmContext';

import {SegmentTitleLinkUtmContextProvider} from '../../contexts/SegmentTitleLinkUtmContext';
import {SegmentBuyButtonUtmContextProvider} from '../../contexts/SegmentBuyButtonUtmContext';
import {SegmentCompanyLinkUtmContextProvider} from '../../contexts/SegmentCompanyLinkUtmContext';

import ServiceDescription from '../ServiceDescription';
import SearchFooter from '../SearchFooter/SearchFooter';
import SearchHeader from '../SearchHeader/SearchHeader.desktop';
import SearchErrorDesktop from '../SearchError/SearchError.desktop';
import SearchSegments from '../SearchSegments/SearchSegments.desktop';
import SearchHeaderDisclaimers from '../SearchHeaderDisclaimers/SearchHeaderDisclaimers';

import keyset from '../../i18n/search-title-notfound';

const b = B('SearchPage');

const segmentCompanyLinkUtmContext = {
    getUtmMedium: transportType => {
        if (transportType !== TransportType.plane) {
            return undefined;
        }

        return 'rasp_airline';
    },
};

export default memo(SearchPage);

function SearchPage(): React.ReactElement {
    const tld = useSelector(state => state.tld);
    const flags = useFlags();
    const language = useSelector(state => state.language);
    const page = useSelector(state => state.page);
    const environment = useSelector(state => state.environment);
    const time = useSelector(state => state.searchForm.time);
    const currencies = useSelector(state => state.currencies);
    const searchCalendar = useSelector(state => state.searchCalendar);
    const search = useSelector(state => state.search);
    const user = useSelector(state => state.user);

    const {
        context,
        sort,
        blablacar,
        querying,
        filtering,
        sortMapping,
        segments,
        transportTypes,
        isSuburbanSearchResult,
        plans,
        archivalData,
        bannerInfo,
    } = search;
    const {browser} = user;
    const hasDescription = shouldShowSearchDescription(search, page);
    const dispatch = useDispatch();
    const queryingPrices = querying.dynamic || querying.train;
    const {
        segments: segmentsMapped = segments,
        vector,
        filteredSegmentIndices = filtering.filteredSegmentIndices,
    } = sortMapping;
    const isAllDays = isAllDaysSearch(context);

    const segmentTitleLinkUtmContext = useMemo(
        () => ({
            getUtmMedium: transportType => {
                if (transportType !== TransportType.plane) {
                    return undefined;
                }

                return isAllDays ? 'rasp_landing_alldays' : 'rasp_landing';
            },
        }),
        [isAllDays],
    );

    if (
        isResultEmpty({segments: segmentsMapped, page}) ||
        archivalData?.hasSegments
    ) {
        return (
            <div className={b()}>
                <SearchErrorDesktop
                    context={context}
                    sort={sort}
                    plans={plans}
                    filtering={filtering}
                    currencies={currencies}
                    time={time}
                    type={SearchErrorPageType.noTickets}
                />

                {archivalData?.hasSegments && (
                    <div className={b('archivalFlightsTitle')}>
                        {keyset('archival-flights-title', {
                            from: context.from.title,
                            to: context.to.title,
                        })}
                    </div>
                )}

                {Boolean(segments.length) && (
                    <SearchSegments
                        flags={flags}
                        vector={buildArchivalSegmentsVector(segmentsMapped)}
                        context={context}
                        blablacar={blablacar}
                        segments={segmentsMapped}
                        querying={querying}
                        filtering={filtering}
                        dispatch={dispatch}
                        language={language}
                        sort={sort}
                        filteredSegmentIndices={filteredSegmentIndices}
                        environment={environment}
                        transportTypes={transportTypes}
                        isSuburbanSearchResult={isSuburbanSearchResult}
                        archivalData={archivalData}
                        browser={browser}
                        bannerInfo={bannerInfo}
                    />
                )}

                {archivalData && <ServiceDescription />}
            </div>
        );
    }

    return (
        <SegmentTitleLinkUtmContextProvider value={segmentTitleLinkUtmContext}>
            <SegmentCompanyLinkUtmContextProvider
                value={segmentCompanyLinkUtmContext}
            >
                <SegmentBuyButtonUtmContextProvider
                    value={valueForSegmentBuyButtonUtmContext}
                >
                    <div className={b()}>
                        <div className={b('content')}>
                            <SearchHeader
                                className={b('header')}
                                time={time}
                                sort={sort}
                                plans={plans}
                                context={context}
                                filtering={filtering}
                                segments={segments}
                                queryingPrices={queryingPrices}
                                hasDescription={hasDescription}
                                transportTypes={transportTypes}
                                currencies={currencies}
                                language={language}
                                tld={tld}
                                searchCalendar={searchCalendar}
                                isSuburbanSearchResult={isSuburbanSearchResult}
                            />

                            <SearchHeaderDisclaimers
                                className={b('disclaimers')}
                                context={context}
                                segments={segments}
                                redesigned
                            />

                            <SearchSegments
                                flags={flags}
                                vector={vector}
                                context={context}
                                blablacar={blablacar}
                                segments={segmentsMapped}
                                querying={querying}
                                filtering={filtering}
                                dispatch={dispatch}
                                language={language}
                                sort={sort}
                                filteredSegmentIndices={filteredSegmentIndices}
                                environment={environment}
                                transportTypes={transportTypes}
                                isSuburbanSearchResult={isSuburbanSearchResult}
                                queryingPrices={queryingPrices}
                                browser={browser}
                                hideAds={user.isBot}
                                bannerInfo={bannerInfo}
                            />

                            <SearchFooter />
                        </div>
                    </div>
                </SegmentBuyButtonUtmContextProvider>
            </SegmentCompanyLinkUtmContextProvider>
        </SegmentTitleLinkUtmContextProvider>
    );
}
