import React, {memo, useMemo} from 'react';
import B from 'bem-cn-lite';

import {FilterTransportType, TransportType} from '../../lib/transportType';
import TeasersRecord from '../../interfaces/state/teasers/TeasersRecord';
import TeaserType from '../../interfaces/state/teasers/TeaserType';
import ICanonicalFromBackend from '../../interfaces/api/ICanonicalFromBackend';
import SearchErrorPageType from '../../../common/interfaces/components/searchErrorPage/SearchErrorPageType';
import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import Slug from '../../interfaces/Slug';

import isResultEmpty from '../../lib/search/isResultEmpty';
import isAllDaysSearch from '../../lib/search/isAllDaysSearch';
import {shouldShowSearchDescription} from '../../lib/seo/search';
import useSelector from '../useSelector';
import useDispatch from '../useDispatch';
import {useFlags} from '../../hooks/useFlags';
import buildArchivalSegmentsVector from '../../lib/sort/buildArchivalSegmentsVector';
import getCanonicalSearchUrl from '../../lib/url/getCanonicalSearchUrl';
import valueForSegmentBuyButtonUtmContext from './utilities/valueForSegmentBuyButtonUtmContext';

import {SegmentCompanyLinkUtmContextProvider} from '../../contexts/SegmentCompanyLinkUtmContext';
import {SegmentBuyButtonUtmContextProvider} from '../../contexts/SegmentBuyButtonUtmContext';
import {SegmentTitleLinkUtmContextProvider} from '../../contexts/SegmentTitleLinkUtmContext';

import Teaser from '../Teaser';
import LinksBlock from '../LinksBlock';
import ServiceDescription from '../ServiceDescription';
import SearchFooter from '../SearchFooter/SearchFooter';
import SegmentBottomSheet from '../search/SegmentBottomSheet';
import SearchHeader from '../SearchHeader/SearchHeader.mobile';
import SearchErrorMobile from '../SearchError/SearchError.mobile';
import SearchSegments from '../SearchSegments/SearchSegments.mobile';
import MobileFooterDirect from './components/MobileFooterDirect/MobileFooterDirect';
import SearchFormDateButtons from '../SearchFormDateButtons/SearchFormDateButtons';
import SearchPopularDirections from '../SearchPopularDirections/SearchPopularDirections';

import otherTransportTypesKeyset from '../../i18n/search-other-transport-types';

const b = B('SearchPage');

function getTeaser(
    type: TeaserType,
    teasers: TeasersRecord,
): React.ReactElement | null {
    const teaser = teasers[type];

    return teaser ? (
        <Teaser className={b('teaser')} type={type} {...teaser} />
    ) : null;
}

const segmentCompanyLinkUtmContext = {
    getUtmMedium: transportType => {
        if (transportType !== TransportType.plane) {
            return undefined;
        }

        return 'rasp_airline';
    },
};

interface IGetOtherTransportTypesParams {
    canonicals: ICanonicalFromBackend[] | null;
    transportType: FilterTransportType;
    tld: Tld;
    language: Lang;
    currentPath: string;
    fromSlug: Slug;
    toSlug: Slug;
}

function getOtherTransportTypes({
    canonicals,
    transportType,
    tld,
    language,
    currentPath,
    fromSlug,
    toSlug,
}: IGetOtherTransportTypesParams): React.ReactElement | null {
    if (!canonicals || !canonicals.length) {
        return null;
    }

    // Показывать блок нужно только для канонических страниц
    if (
        currentPath !==
        getCanonicalSearchUrl({transportType, tld, fromSlug, toSlug, language})
    ) {
        return null;
    }

    const title =
        transportType === FilterTransportType.all
            ? otherTransportTypesKeyset('by-transport-types')
            : otherTransportTypesKeyset('other-transport-types');

    const linksOnlyOnOtherTransportTypes = canonicals.filter(canonical => {
        if (
            canonical.canonical.transportType === null &&
            transportType !== FilterTransportType.all
        ) {
            return true;
        }

        if (
            canonical.canonical.transportType !== null &&
            FilterTransportType[canonical.canonical.transportType] !==
                transportType
        ) {
            return true;
        }

        return false;
    });

    if (linksOnlyOnOtherTransportTypes.length <= 1) {
        return null;
    }

    const links = linksOnlyOnOtherTransportTypes.map(canonical => {
        const keysetParams = {
            fromTitle: canonical.pointFrom.title,
            toTitle: canonical.pointTo.title,
        };

        let linksText = '';

        switch (canonical.canonical.transportType) {
            case null:
                linksText = otherTransportTypesKeyset('all', keysetParams);
                break;
            case TransportType.train:
                linksText = otherTransportTypesKeyset('train', keysetParams);
                break;
            case TransportType.suburban:
                linksText = otherTransportTypesKeyset('suburban', keysetParams);
                break;
            case TransportType.plane:
                linksText = otherTransportTypesKeyset('plane', keysetParams);
                break;
            case TransportType.water:
                linksText = otherTransportTypesKeyset('water', keysetParams);
                break;
            case TransportType.bus:
                linksText = otherTransportTypesKeyset('bus', keysetParams);
                break;
        }

        return {
            href: getCanonicalSearchUrl({
                fromSlug: canonical.canonical.pointFrom,
                toSlug: canonical.canonical.pointTo,
                transportType:
                    canonical.canonical.transportType === null
                        ? FilterTransportType.all
                        : FilterTransportType[
                              canonical.canonical.transportType
                          ],
                tld,
                language,
            }),
            text: linksText,
        };
    });

    return (
        <LinksBlock className={b('otherTypes')} title={title} links={links} />
    );
}

export default memo(SearchPage);

function SearchPage(): React.ReactElement {
    const flags = useFlags();
    const language = useSelector(state => state.language);
    const page = useSelector(state => state.page);
    const environment = useSelector(state => state.environment);
    const searchForm = useSelector(state => state.searchForm);
    const currencies = useSelector(state => state.currencies);
    const search = useSelector(state => state.search);
    const tld = useSelector(state => state.tld);
    const user = useSelector(state => state.user);
    const withDateButtons = isAllDaysSearch(searchForm);

    const {
        context,
        sort,
        blablacar,
        querying,
        filtering,
        sortMapping,
        segments,
        transportTypes,
        isSuburbanSearchResult,
        teasers,
        archivalData,
        canonicals,
        bannerInfo,
    } = search;
    const {browser} = user;

    const hasDescription = shouldShowSearchDescription(search, page);

    const dispatch = useDispatch();

    const {
        segments: segmentsMapped = segments,
        vector,
        filteredSegmentIndices = filtering.filteredSegmentIndices,
    } = sortMapping;
    const isAllDays = isAllDaysSearch(context);

    const segmentTitleLinkUtmContext = useMemo(
        () => ({
            getUtmMedium: transportType => {
                if (transportType !== TransportType.plane) {
                    return undefined;
                }

                return isAllDays ? 'rasp_landing_alldays' : 'rasp_landing';
            },
        }),
        [isAllDays],
    );

    if (
        isResultEmpty({segments: segmentsMapped, page}) ||
        archivalData?.hasSegments
    ) {
        return (
            <div className={b()}>
                {withDateButtons && <SearchFormDateButtons />}

                <SearchErrorMobile
                    context={context}
                    sort={sort}
                    type={SearchErrorPageType.noTickets}
                />

                {Boolean(segments.length) && (
                    <SearchSegments
                        flags={flags}
                        vector={buildArchivalSegmentsVector(segmentsMapped)}
                        context={context}
                        blablacar={blablacar}
                        segments={segmentsMapped}
                        querying={querying}
                        filtering={filtering}
                        dispatch={dispatch}
                        language={language}
                        sort={sort}
                        filteredSegmentIndices={filteredSegmentIndices}
                        environment={environment}
                        transportTypes={transportTypes}
                        isSuburbanSearchResult={isSuburbanSearchResult}
                        archivalData={archivalData}
                        browser={browser}
                        bannerInfo={bannerInfo}
                    />
                )}

                {archivalData && <ServiceDescription />}
            </div>
        );
    }

    const getOtherTransportTypesParams = {
        canonicals,
        transportType: context.transportType,
        tld,
        language,
        currentPath: page.location.path,
        fromSlug: context.originalFrom.slug,
        toSlug: context.originalTo.slug,
    };

    return (
        <SegmentTitleLinkUtmContextProvider value={segmentTitleLinkUtmContext}>
            <SegmentCompanyLinkUtmContextProvider
                value={segmentCompanyLinkUtmContext}
            >
                <SegmentBuyButtonUtmContextProvider
                    value={valueForSegmentBuyButtonUtmContext}
                >
                    <div className={b()}>
                        <div className={b('content')}>
                            {withDateButtons && <SearchFormDateButtons />}

                            <div className={b('teasers')}>
                                {getTeaser(TeaserType.attention, teasers)}
                                {getTeaser(TeaserType.normal, teasers)}
                                {getTeaser(TeaserType.special, teasers)}
                                {getTeaser(TeaserType.banner, teasers)}
                            </div>

                            <SearchHeader
                                context={context}
                                segments={segmentsMapped}
                                hasDescription={hasDescription}
                            />

                            <SearchSegments
                                flags={flags}
                                vector={vector}
                                context={context}
                                blablacar={blablacar}
                                segments={segmentsMapped}
                                querying={querying}
                                filtering={filtering}
                                dispatch={dispatch}
                                language={language}
                                sort={sort}
                                filteredSegmentIndices={filteredSegmentIndices}
                                environment={environment}
                                transportTypes={transportTypes}
                                isSuburbanSearchResult={isSuburbanSearchResult}
                                browser={browser}
                                bannerInfo={bannerInfo}
                            />

                            <MobileFooterDirect page={page} flags={flags} />

                            {getOtherTransportTypes(
                                getOtherTransportTypesParams,
                            )}

                            <SearchPopularDirections
                                className={b('popularDirections')}
                            />

                            <SearchFooter
                                className={b('footer')}
                                withHelpLinks
                            />

                            <SegmentBottomSheet
                                sort={sort}
                                context={context}
                                currencies={currencies}
                                isSuburbanSearchResult={isSuburbanSearchResult}
                            />
                        </div>
                    </div>
                </SegmentBuyButtonUtmContextProvider>
            </SegmentCompanyLinkUtmContextProvider>
        </SegmentTitleLinkUtmContextProvider>
    );
}
