import React, {PureComponent, createRef} from 'react';
import B from 'bem-cn-lite';
import {connect} from 'react-redux';

import {HEADER_HEIGHT} from '../Header/Header.desktop';

import Platform from '../../interfaces/Platform';
import IStateFlags from '../../interfaces/state/flags/IStateFlags';
import IStatePage from '../../interfaces/state/IStatePage';
import TeasersRecord from '../../interfaces/state/teasers/TeasersRecord';
import MapStateToProps from '../../interfaces/MapStateToProps';

import isHTMLElement from '../../lib/isHTMLElement';

import Teaser from '../Teaser';
import Direct from '../Direct/Direct';
import SearchPopularDirections from '../SearchPopularDirections/SearchPopularDirections';

const b = B('SearchPageSidebar');

enum Position {
    top = 'top', // Блок покоится наверху бокового контейнера
    topToMiddle = 'topToMiddle', // Позиция для анимации
    middleToTop = 'middleToTop', // Позиция для анимации
    bottom = 'bottom', // Блок покоится на дне бокового контейнера
    middleStickToBottom = 'middleStickToBottom', // Блок прибит низом к низу окна
    middleStickToTop = 'middleStickToTop', // Блок прибит верхом к верху окна
    middleScrollable = 'middleScrollable', // Блок находится в подвешенном состоянии посреди страницы
}

const ANIMATION_TIME = 300;
const DEFAULT_DIRECT_HEIGHT = 200;
const MAX_MEDIA_DIRECT_HEIGHT = 600;
const FLOAT_BLOCK_BOTTOM_OFFSET = 20;
const NEXT_POSITIONS = {
    [Position.middleToTop]: Position.top,
    [Position.topToMiddle]: Position.middleStickToTop,
};

const TOP_DIRECT_ID = 'R-I-94177-54';
const BOTTOM_DIRECT_ID = 'R-I-94177-55';

const isDesktop = process.env.PLATFORM === Platform.desktop;

interface ISearchPageSidebarState {
    position: Position;
    showBottomDirectBlock: boolean;
    isMedia: boolean;

    directHeight?: number;
    directBlockStyle?: {width: number};
}

interface ISearchPageSidebarPropsFromState {
    page: IStatePage;
}

interface ISearchPageSidebarProps extends ISearchPageSidebarPropsFromState {
    flags: IStateFlags;
    teasers?: TeasersRecord;
}

enum ScrollDirection {
    up = 'up',
    down = 'down',
}

interface IBlockPosition {
    topBorder: number;
    bottomBorder: number;
    sidebarOffsetTop: number;
    floatBlockToClientTop: number;
    floatBlockToClientBottom: number;
    floatBlockToSidebarBottom: number;
    floatBlockToFloatHolderBottom: number;
}

interface IParams {
    sidebarPaddingTop: number;
    pageYOffset: number;
    directOffsetTop: number;

    debounceTm?: number;
}

const mapStateToProps: MapStateToProps<ISearchPageSidebarPropsFromState> = ({
    page,
}) => ({page});

class SearchPageSidebar extends PureComponent<
    ISearchPageSidebarProps,
    ISearchPageSidebarState
> {
    sidebar = createRef<HTMLDivElement>();
    floatHolder = createRef<HTMLDivElement>();
    floatBlock = createRef<HTMLDivElement>();
    params: IParams = {
        sidebarPaddingTop: 0,
        pageYOffset: 0,
        directOffsetTop: 0,
    };

    waitingScrollHanding = false;

    constructor(props) {
        super(props);

        this.state = {
            position: Position.top,
            directHeight: 0,
            showBottomDirectBlock: false,
            isMedia: false,
        };
    }

    componentDidMount(): void {
        window.addEventListener('scroll', this.requestScrollHandler);
    }

    componentWillUnmount(): void {
        window.removeEventListener('scroll', this.requestScrollHandler);
    }

    onTopDirectBlockRender = ({ref}): null | void => {
        if (!ref.current) {
            return null;
        }

        const {height} = ref.current.getBoundingClientRect();

        const isMedia = height <= MAX_MEDIA_DIRECT_HEIGHT;

        this.setState({
            isMedia,
        });

        this.setState({showBottomDirectBlock: isMedia});
    };

    getBottomDirect(): React.ReactElement | null {
        const {
            flags: {idDirectSearchRightBottomDesktop},
        } = this.props;

        const bottomDirectId =
            idDirectSearchRightBottomDesktop || BOTTOM_DIRECT_ID;

        return (
            <Direct
                className={b('directBottomBlock')}
                blockId={bottomDirectId}
            />
        );
    }

    getCurrentPositions(): IBlockPosition | null {
        const sidebarElement = this.sidebar && this.sidebar.current;
        const floatHolder = this.floatHolder && this.floatHolder.current;
        const floatBlock = this.floatBlock && this.floatBlock.current;

        if (!sidebarElement || !floatHolder || !floatBlock) {
            return null;
        }

        const sidebarClientRect = sidebarElement.getBoundingClientRect();
        const floatHolderClientRect = floatHolder.getBoundingClientRect();
        const floatBlockClientRect = floatBlock.getBoundingClientRect();

        if (
            !sidebarClientRect ||
            !floatHolderClientRect ||
            !floatBlockClientRect
        ) {
            return null;
        }

        this.params.sidebarPaddingTop =
            this.params.sidebarPaddingTop ||
            parseInt(window.getComputedStyle(sidebarElement).paddingTop, 10);

        const sidebarHeight =
            sidebarElement.clientHeight - this.params.sidebarPaddingTop * 2;
        const bottomBorder =
            sidebarHeight - floatBlock.offsetHeight - HEADER_HEIGHT;

        // Дистанция между нижним краем экрана и низом рекламного блока
        const floatBlockToClientBottom =
            window.innerHeight - floatBlockClientRect.bottom;
        // Дистанция между верхним краем экрана и верхом рекламного блока
        const floatBlockToClientTop = floatBlockClientRect.top * -1;
        // Дистанция между низом контейнера-правой колонки
        const floatBlockToSidebarBottom =
            sidebarClientRect.bottom -
            floatBlockClientRect.bottom -
            FLOAT_BLOCK_BOTTOM_OFFSET;
        // Дистанция между низом стартовой позиции рекламеного блока и низом рекламного блока
        const floatBlockToFloatHolderBottom =
            floatHolderClientRect.bottom - floatBlockClientRect.bottom;

        let topBorder = -HEADER_HEIGHT;

        Array.from(sidebarElement.children).forEach(child => {
            if (isHTMLElement(child)) {
                topBorder += child.offsetHeight;
            }
        });
        topBorder += this.params.sidebarPaddingTop;

        // разница между экраном и верхней позицией сайдбара
        const sidebarOffsetTop = sidebarClientRect.top * -1;

        return {
            topBorder,
            bottomBorder,
            sidebarOffsetTop,
            floatBlockToClientBottom,
            floatBlockToClientTop,
            floatBlockToSidebarBottom,
            floatBlockToFloatHolderBottom,
        };
    }

    getPositionWithTopStickLogic(
        currentPosition: Position,
        topBorder: number,
        bottomBorder: number,
        sidebarOffsetTop: number,
    ): Position | null {
        // Иногда реклама меняется с не-медийки на медийку без предупреждения, поэтому нужно переключаться тут
        if (
            currentPosition === Position.middleStickToBottom ||
            currentPosition === Position.middleScrollable
        ) {
            return Position.middleStickToTop;
        }

        if (sidebarOffsetTop > topBorder) {
            if (sidebarOffsetTop > bottomBorder) {
                return Position.bottom;
            }

            if (currentPosition === Position.bottom) {
                return Position.middleStickToTop;
            }

            if (currentPosition === Position.top) {
                return Position.topToMiddle;
            }

            return null;
        }

        if (currentPosition !== Position.top) {
            if (currentPosition !== Position.middleToTop) {
                return Position.middleToTop;
            }

            if (sidebarOffsetTop < 0) {
                return Position.top;
            }

            return null;
        }

        return null;
    }

    getNewDirectPosition(
        {
            topBorder,
            bottomBorder,
            sidebarOffsetTop,
            floatBlockToClientBottom,
            floatBlockToClientTop,
            floatBlockToSidebarBottom,
            floatBlockToFloatHolderBottom,
        }: IBlockPosition,
        direction: ScrollDirection,
    ): Position | null {
        const currentPosition = this.state.position;
        const {isMedia} = this.state;

        const edgeStickLogic = !isMedia;
        const bottomOfFloatBlockIsAboveTheBottomOfScreen =
            floatBlockToClientBottom >= FLOAT_BLOCK_BOTTOM_OFFSET;

        if (edgeStickLogic) {
            // Расстояние между верхом рекламного блока и верхним краем сайдбара
            const newDirectOffsetTop = sidebarOffsetTop - floatBlockToClientTop;

            switch (currentPosition) {
                case Position.top: {
                    if (bottomOfFloatBlockIsAboveTheBottomOfScreen) {
                        return Position.middleStickToBottom;
                    }

                    return null;
                }

                case Position.middleStickToTop: {
                    if (direction === ScrollDirection.down) {
                        this.params.directOffsetTop = newDirectOffsetTop;

                        return Position.middleScrollable;
                    }

                    // Если доскроллили до верху, прикрепляем рекламу наверх сайдбара
                    if (floatBlockToFloatHolderBottom >= 0) {
                        return Position.top;
                    }

                    return null;
                }

                case Position.middleScrollable: {
                    // Если скроллим вниз и нижняя граница рекламы оказывается выше чем нижний паддинг, то прикрепляем рекламу к низу экрана
                    if (
                        floatBlockToClientBottom > FLOAT_BLOCK_BOTTOM_OFFSET &&
                        direction === ScrollDirection.down
                    ) {
                        return Position.middleStickToBottom;
                    }

                    // Если скроллим наверх и верхняя граница рекламы оказывается ниже чем хедер+паддинг, то прикрепляем рекламу к верху экрана
                    if (
                        floatBlockToClientTop +
                            this.params.sidebarPaddingTop +
                            HEADER_HEIGHT <
                            0 &&
                        direction === ScrollDirection.up
                    ) {
                        return Position.middleStickToTop;
                    }

                    return null;
                }

                case Position.middleStickToBottom: {
                    if (direction === ScrollDirection.up) {
                        this.params.directOffsetTop = newDirectOffsetTop;

                        return Position.middleScrollable;
                    }

                    // Если доскроллили до низу, прикрепляем рекламу вниз сайдбара
                    if (floatBlockToSidebarBottom < 0) {
                        return Position.bottom;
                    }

                    return null;
                }

                case Position.bottom: {
                    // Если скроллим наверх и расстояние от верха экрана до верха сайдбара находится
                    // между положениями "верх прикреплённого к низу сайдбара блока" и "низ прикреплённого к верху сайдбара блока",
                    // то прикрепляем к верху экрана
                    if (
                        sidebarOffsetTop > topBorder &&
                        sidebarOffsetTop <= bottomBorder &&
                        direction === ScrollDirection.up
                    ) {
                        return Position.middleStickToTop;
                    }

                    return null;
                }
            }

            return null;
        }

        return this.getPositionWithTopStickLogic(
            currentPosition,
            topBorder,
            bottomBorder,
            sidebarOffsetTop,
        );
    }

    requestScrollHandler = (): void => {
        if (this.waitingScrollHanding) {
            return;
        }

        this.waitingScrollHanding = true;
        requestAnimationFrame(() => {
            this.scrollHandler();
            this.waitingScrollHanding = false;
        });
    };

    scrollHandler = (): null | void => {
        const oldYOffset = this.params.pageYOffset;
        const newYOffset = window.pageYOffset;
        const yOffset = newYOffset - oldYOffset;

        if (!yOffset) {
            return null;
        }

        const direction =
            newYOffset - oldYOffset > 0
                ? ScrollDirection.down
                : ScrollDirection.up;

        this.params.pageYOffset = newYOffset;

        const currentPositions = this.getCurrentPositions();

        if (!currentPositions) {
            return null;
        }

        const directPosition = this.getNewDirectPosition(
            currentPositions,
            direction,
        );

        if (directPosition) {
            this.changeDirectPosition(directPosition);
        }
    };

    changeDirectPosition(position: Position): void {
        const nextPosition = NEXT_POSITIONS[position];

        if (
            this.params.debounceTm &&
            (position === Position.top || Position.bottom)
        ) {
            clearTimeout(this.params.debounceTm);
        }

        this.changeFloatState(position);

        if (nextPosition) {
            this.params.debounceTm = window.setTimeout(() => {
                this.changeFloatState(nextPosition);
            }, ANIMATION_TIME);
        }
    }

    changeFloatState(newPosition: Position): void {
        this.setState({
            position: newPosition,
            directHeight: this.floatBlock?.current?.offsetHeight,
        });
    }

    getFloatClass(): string {
        return b('floatblock', {
            position: this.state.position,
        });
    }

    getTeaser(type): React.ReactElement | null {
        const {teasers = {}} = this.props;

        if (!isDesktop || !teasers[type]) {
            return null;
        }

        return <Teaser type={type} {...teasers[type]} />;
    }

    render(): React.ReactElement {
        const {showBottomDirectBlock, isMedia} = this.state;
        const floatHolderStyles = {
            minHeight: this.state.directHeight || DEFAULT_DIRECT_HEIGHT,
        };
        const floatBlockStyles =
            this.state.position === Position.middleScrollable
                ? {top: this.params.directOffsetTop}
                : undefined;

        const {
            flags: {idDirectSearchRightTopDesktop},
        } = this.props;

        const topDirectId = idDirectSearchRightTopDesktop || TOP_DIRECT_ID;

        return (
            <div className={b()} ref={this.sidebar}>
                {this.getTeaser('attention')}
                {this.getTeaser('special')}
                {this.getTeaser('normal')}
                {this.getTeaser('banner')}

                <div
                    className={b('floatHolder')}
                    style={floatHolderStyles}
                    ref={this.floatHolder}
                >
                    <div
                        className={this.getFloatClass()}
                        style={floatBlockStyles}
                        ref={this.floatBlock}
                    >
                        <Direct
                            className={b('directTopBlock')}
                            blockId={topDirectId}
                            onRender={this.onTopDirectBlockRender}
                        />

                        {showBottomDirectBlock && this.getBottomDirect()}

                        {!isMedia && (
                            <SearchPopularDirections
                                className={b('directions')}
                            />
                        )}
                    </div>
                </div>

                {isMedia && <SearchPopularDirections />}
            </div>
        );
    }
}

export default connect(mapStateToProps)(SearchPageSidebar);
