import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';
import isEmpty from 'lodash/isEmpty';
import {connect} from 'react-redux';

import {CHAR_EM_DASH} from '../../lib/stringUtils';

import {isCity} from '../../lib/point/pointType';
import getTariffKeys from '../../lib/segments/getTariffKeys';
import isTransferSegmentFunc from '../../lib/segments/isTransferSegment';
import getSubtypeByTransportType from '../../lib/station/getSubtypeByTransportType';
import {getActualBadgesToDisplayAfterFiltering} from './utilities/getActualBadgesToDisplayAfterFiltering';

import Point from './components/Point/Point';
import Station from './components/Station/Station';
import SegmentBadges from '../SegmentBadges/SegmentBadges';
import SegmentHeader from './components/SegmentHeader/SegmentHeader';
import TimeAndStations from './components/TimeAndStations/TimeAndStations';
import ScheduleAndPrices from './components/ScheduleAndPrices/ScheduleAndPrices';
import IntervalTimeAndStations from './components/IntervalTimeAndStations/IntervalTimeAndStations';

const b = B('SearchSegment');

const TOP_OFFSET = 16;
const TOP_OFFSET_WITH_BADGE = 29;
const TOP_TRANSFER_OFFSET = 10;

const mapStateToProps = ({flags, language, currencies, seoQueryParams}) => ({
    flags,
    language,
    currencies,
    seoQueryParams,
});

class SearchSegmentDesktop extends PureComponent {
    constructor(props) {
        super(props);

        this.parentRef = React.createRef();
        this.departureChildrenRef = React.createRef();
        this.arrivalChildrenRef = React.createRef();

        this.state = {
            departureTop: 0,
            arrivalTop: 0,
        };
    }

    componentDidUpdate() {
        /*
            Считаем отступ сверху до станции прибытия и отпраления (из 2 и 4 столбцов) чтобы наложить через position: absolute
            станции прибытия и отправления из последнего столбца, а первые скрыть. Это нужно для СЕО, и при этом чтобы и людям и
            роботам отдавалась одинаковая верстка.
        */
        if (
            this.parentRef.current &&
            this.departureChildrenRef.current &&
            this.arrivalChildrenRef.current
        ) {
            const departureTop =
                this.departureChildrenRef.current.getBoundingClientRect().top -
                this.parentRef.current.getBoundingClientRect().top;
            const arrivalTop =
                this.arrivalChildrenRef.current.getBoundingClientRect().top -
                this.parentRef.current.getBoundingClientRect().top;

            if (
                arrivalTop !== this.state.arrivalTop ||
                departureTop !== this.state.departureTop
            ) {
                this.setState({
                    departureTop,
                    arrivalTop,
                });
            }
        }
    }

    render() {
        const {
            data,
            children,
            className,
            seoQueryParams,
            flags,
            withActionButton = true,
            withoutPrices = false,
            language,
            currencies,
        } = this.props;

        const {departureTop, arrivalTop} = this.state;

        const {
            context,
            sort,
            currentTimeMoment,
            isSuburbanSearchResult,
            isHidden,
            showStops = true,
            isStationsVisible = true,
            showArrivalDate = false,
            showDepartureDate = false,
            showTransportIcon = true,
            isSegmentOfInterline,
            segment,
        } = data;
        const {
            queryingPrices = false,
            stationFrom,
            stationTo,
            useCityInsteadStationTo,
            useCityInsteadStationFrom,
            transport,
            thread,
        } = segment;

        const {from, to} = context;
        const transportType = transport.code;
        const stationSubtype = getSubtypeByTransportType(transportType);

        const isTransferSegment = isTransferSegmentFunc(segment);
        const badges = getActualBadgesToDisplayAfterFiltering(segment);
        const cancelType =
            'cancelType' in segment ? segment.cancelType : undefined;
        const {mainTariffKey: klass, tariffKeys: tariffClassKeys} =
            getTariffKeys(segment, flags, seoQueryParams, sort);

        const isNotEmptyBadges = !isEmpty(badges) || Boolean(cancelType);
        const mods = {
            isNotEmptyBadges,
            isHidden,
            isTransferSegment,
            isMetaSegment: segment.isMetaSegment,
        };

        const snippetOffset = isNotEmptyBadges
            ? TOP_OFFSET_WITH_BADGE
            : TOP_OFFSET;
        const topOffset = isTransferSegment
            ? TOP_TRANSFER_OFFSET
            : snippetOffset;
        const showPrices = !isSegmentOfInterline && !segment.isArchival;

        return (
            <tr className={b(mods, className)} ref={this.parentRef}>
                <SegmentHeader
                    className={b('header')}
                    context={context}
                    segment={segment}
                    tariffClassKeys={tariffClassKeys}
                    currentTimeMoment={currentTimeMoment}
                    showTransportIcon={showTransportIcon}
                    isSuburbanSearchResult={isSuburbanSearchResult}
                />

                {segment.isInterval ? (
                    <IntervalTimeAndStations
                        segment={segment}
                        flags={flags}
                        language={language}
                        isStationsVisible={isStationsVisible}
                        departureChildrenRef={this.departureChildrenRef}
                        arrivalChildrenRef={this.arrivalChildrenRef}
                        stationOpacity={departureTop > 0 ? 0 : 1}
                    />
                ) : (
                    <TimeAndStations
                        flags={flags}
                        language={language}
                        context={context}
                        segment={segment}
                        showStops={showStops}
                        isStationsVisible={isStationsVisible}
                        showDepartureDate={showDepartureDate}
                        showArrivalDate={showArrivalDate}
                        departureChildrenRef={this.departureChildrenRef}
                        arrivalChildrenRef={this.arrivalChildrenRef}
                        stationOpacity={departureTop > 0 ? 0 : 1}
                    />
                )}

                {showPrices && !withoutPrices && (
                    <ScheduleAndPrices
                        tag="td"
                        segment={segment}
                        context={context}
                        isSuburbanSearchResult={isSuburbanSearchResult}
                        queryingPrices={queryingPrices}
                        tariffClassKeys={tariffClassKeys}
                        klass={klass}
                        withActionButton={withActionButton}
                        flags={flags}
                        language={language}
                        currencies={currencies}
                    />
                )}

                <td className={b('badges')}>
                    <SegmentBadges
                        badges={badges}
                        cancelType={cancelType}
                        cancelledSegments={thread && thread.cancelledSegments}
                    />
                </td>

                {segment.isInterval ? (
                    <td className={b('stationColumn', {interval: true})}>
                        <div
                            className={b('intervalStations', {
                                visible: departureTop > 0 ? 1 : 'none',
                            })}
                            style={{top: departureTop - topOffset}}
                        >
                            <Station
                                className={b('station')}
                                station={stationFrom}
                                stationIsVisible={isStationsVisible}
                                stationSubtype={stationSubtype}
                            />
                            {` ${CHAR_EM_DASH} `}
                            <Station
                                className={b('station')}
                                station={stationTo}
                                stationIsVisible={isStationsVisible}
                                stationSubtype={stationSubtype}
                            />
                        </div>
                    </td>
                ) : (
                    <td
                        className={b('stationColumn', {
                            transfer: isTransferSegment,
                        })}
                    >
                        <div
                            className={b('stationPoint', {
                                visible: departureTop > 0 ? 1 : 'none',
                            })}
                            style={{top: departureTop - topOffset}}
                        >
                            <Point
                                b={b}
                                station={stationFrom}
                                city={isCity(from) ? from : null}
                                stationIsVisible={isStationsVisible}
                                useCityInsteadStation={
                                    useCityInsteadStationFrom
                                }
                                stationSubtype={stationSubtype}
                            />
                        </div>

                        <div
                            className={b('stationPoint', {
                                arrival: true,
                                visible: arrivalTop > 0 ? 1 : 'none',
                            })}
                            style={{top: arrivalTop - topOffset}}
                        >
                            <Point
                                b={b}
                                station={stationTo}
                                city={isCity(to) ? to : null}
                                stationIsVisible={isStationsVisible}
                                useCityInsteadStation={useCityInsteadStationTo}
                                stationSubtype={stationSubtype}
                            />
                        </div>
                    </td>
                )}

                {children}
            </tr>
        );
    }
}

export default connect(mapStateToProps)(SearchSegmentDesktop);
