import B from 'bem-cn-lite';
import React, {useEffect, useContext, useCallback} from 'react';
import {publish} from 'pubsub-js';

import {SEGMENTS} from '../../lib/constants/pubSubEvents';

import {TransportType} from '../../lib/transportType';
import ISegment from '../../interfaces/segment/ISegment';
import IconGlyph from '../../interfaces/components/IconGlyph';
import ISegmentBadges from '../../interfaces/segment/ISegmentBadges';
import ISearchSegment from '../../interfaces/components/ISearchSegment';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';

import useSelector from '../useSelector';
import {useFlags} from '../../hooks/useFlags';
import {isCity} from '../../lib/point/pointType';
import {reachGoalOnce} from '../../lib/yaMetrika';
import isSegment from '../../lib/segments/isSegment';
import getTariffKeys from '../../lib/segments/getTariffKeys';
import isAllDaysSearchFunc from '../../lib/search/isAllDaysSearch';
import getThreadUrlForSegment from '../../lib/segments/getThreadUrlForSegment';
import getEndpointsData from '../../lib/segments/endpointEvents/getEndpointsData';
import {
    getLocalDepartureTime,
    getLocalArrivalTime,
} from '../../lib/segments/cachedSegmentTime';
import {getActualBadgesToDisplayAfterFiltering} from './utilities/getActualBadgesToDisplayAfterFiltering';

import Icon from '../Icon/Icon';
import SearchPricesBlock from '../SearchPricesBlock';
import SegmentStops from '../SegmentStops/SegmentStops';
import SegmentRunDays from '../SegmentRunDays/SegmentRunDays';
import SegmentTitle from '../SegmentTitle/SegmentTitle.mobile';
import SegmentLabel from './components/SegmentLabel/SegmentLabel';
import SegmentTitleLinkUtmContext from '../../contexts/SegmentTitleLinkUtmContext';
import SegmentTimeAndStations from '../SegmentTimeAndStations/SegmentTimeAndStations.mobile';

import segmentKeyset from '../../i18n/segment';

const b = B('SearchSegment');

const SearchSegment: React.FC<ISearchSegment<ISegment | ITransferSegment>> = ({
    data: {
        segment,
        sort,
        context,
        currentTimeMoment,
        isSuburbanSearchResult,
        isHidden,

        showArrivalDate = false,
        showDepartureDate = false,
        showTransportIcon = true,
    },
    children,
    className,
    withoutSegmentLabel = false,
    withoutPrices = false,
    withActionButton = true,
}) => {
    const seoQueryParams = useSelector(state => state.seoQueryParams);
    const flags = useFlags();
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const clientId = useSelector(state => state.user.clientId);
    const isProduction = useSelector(state => state.environment.production);

    const {getUtmMedium} = useContext(SegmentTitleLinkUtmContext);

    const {
        isGone,
        isInterval,
        transport,
        title,
        thread,
        company,
        number,
        isThroughTrain,
        hasDynamicPricing,
        departureMoment = getLocalDepartureTime(segment),
        arrivalMoment = getLocalArrivalTime(segment),
        departureRailwayMoment,
        arrivalRailwayMoment,
        stationFrom,
        stationTo,
    } = segment;

    const {code: transportType} = transport;
    const {from, to} = context;
    const {timezone} = from;

    const isAllDaysSearch = isAllDaysSearchFunc(context);

    const isExpress = thread ? thread.isExpress : false;
    const useCityInsteadStationFrom =
        'useCityInsteadStationFrom' in segment
            ? segment.useCityInsteadStationFrom
            : false;
    const useCityInsteadStationTo =
        'useCityInsteadStationTo' in segment
            ? segment.useCityInsteadStationTo
            : false;
    const companies = 'companies' in segment ? segment.companies : undefined;
    const suburbanFacilities =
        'suburbanFacilities' in segment ? segment.suburbanFacilities || [] : [];
    const maxArrival = 'maxArrival' in segment ? segment.maxArrival : undefined;
    const minArrival = 'minArrival' in segment ? segment.minArrival : undefined;
    const isFuzzyFrom =
        'isFuzzyFrom' in segment ? segment.isFuzzyFrom : undefined;
    const isFuzzyTo = 'isFuzzyTo' in segment ? segment.isFuzzyTo : undefined;
    const beginTime = (thread && thread.beginTime) || undefined;
    const endTime = (thread && thread.endTime) || undefined;
    const commentExists = Boolean(thread && thread.comment);
    const duration = 'duration' in segment ? segment.duration : undefined;
    const isMerged = 'isMerged' in segment ? segment.isMerged || false : false;
    const runDays = 'runDays' in segment ? segment.runDays : undefined;
    const daysByTimezone =
        'daysByTimezone' in segment ? segment.daysByTimezone : undefined;
    const electronicTicket = Boolean(
        segment.tariffs && segment.tariffs.electronicTicket,
    );
    const cityFrom = isCity(from) ? from : undefined;
    const cityTo = isCity(to) ? to : undefined;
    const stops = 'stops' in segment ? segment.stops : undefined;
    const codeshares = 'codeshares' in segment ? segment.codeshares : undefined;
    const cancelType = 'cancelType' in segment ? segment.cancelType : undefined;

    const density = (thread && thread.density) || '';

    const badges: ISegmentBadges | undefined =
        getActualBadgesToDisplayAfterFiltering(segment);

    const {mainTariffKey} = getTariffKeys(segment, flags, seoQueryParams, sort);

    const endpointData = getEndpointsData(segment);

    const isArchival = isSegment(segment) && segment.isArchival;
    const threadUrl = getThreadUrlForSegment({
        segment,
        tld,
        language,
        seoQueryParams,
        clientId,
        utmMedium: getUtmMedium(transportType),
        flags,
        isProduction,
        isToCitySearchContext: isCity(to),
    });

    useEffect(() => {
        if (commentExists) {
            reachGoalOnce('segment_comment_shown');
        }
    }, [commentExists]);

    const onClickComment = useCallback(() => {
        publish(SEGMENTS.SEGMENT_INFO_CLICK, {segment, klass: mainTariffKey});
    }, [segment, mainTariffKey]);

    const showSearchPricesBlock = !withoutPrices && !isArchival;

    if (isHidden) {
        return null;
    }

    return (
        <article
            className={b(
                {
                    isGone,
                    isCancel: Boolean(cancelType),
                },
                className,
            )}
        >
            <div className={b('label')}>
                {!withoutSegmentLabel && (
                    <SegmentLabel
                        className={b('labelBlock')}
                        isGone={isGone}
                        isInterval={isInterval}
                        badges={badges || {}}
                        endpointData={endpointData}
                        density={density}
                        currentTimeMoment={currentTimeMoment}
                        departureMoment={departureMoment}
                        cancelType={cancelType}
                        cancelledSegments={thread && thread.cancelledSegments}
                    />
                )}
            </div>

            <SegmentTitle
                transport={transport}
                title={title}
                isSuburbanSearchResult={isSuburbanSearchResult}
                number={number}
                showTransportIcon={showTransportIcon}
                thread={thread}
                isThroughTrain={isThroughTrain}
                company={company || undefined}
                companies={companies}
                electronicTicket={electronicTicket}
                hasDynamicPricing={hasDynamicPricing}
                threadUrl={threadUrl}
                facilities={suburbanFacilities}
                showTitle={isArchival}
                codeshares={codeshares}
            />

            <SegmentTimeAndStations
                className={b('timeAndStations')}
                transportType={transportType}
                isExpress={isExpress}
                isInterval={isInterval}
                departureMoment={departureMoment}
                arrivalMoment={arrivalMoment}
                stationFrom={stationFrom}
                stationTo={stationTo}
                showArrivalDate={showArrivalDate}
                showDepartureDate={showDepartureDate}
                isAllDaysSearch={isAllDaysSearch}
                duration={duration}
                beginTime={beginTime}
                endTime={endTime}
                maxArrival={maxArrival}
                minArrival={minArrival}
                isFuzzyFrom={isFuzzyFrom}
                isFuzzyTo={isFuzzyTo}
                useCityInsteadStationTo={useCityInsteadStationTo}
                useCityInsteadStationFrom={useCityInsteadStationFrom}
                cityFrom={cityFrom}
                cityTo={cityTo}
                departureRailwayMoment={departureRailwayMoment || undefined}
                arrivalRailwayMoment={arrivalRailwayMoment || undefined}
            />

            {isAllDaysSearch && !isArchival && (
                <SegmentRunDays
                    isMerged={isMerged}
                    className={b('runDays')}
                    toDay={currentTimeMoment}
                    timezone={timezone}
                    runDays={runDays}
                    daysByTimezone={daysByTimezone}
                />
            )}

            {transportType === TransportType.suburban &&
                Boolean(stops) &&
                stops !== undefined && (
                    <SegmentStops
                        className={b('stops')}
                        stops={stops}
                        onlyStops
                    />
                )}

            {commentExists && (
                <div className={b('comment')} onClick={onClickComment}>
                    <Icon
                        glyph={IconGlyph.comment2}
                        className={b('commentIcon')}
                    />
                    {segmentKeyset('carrier-comment')}
                </div>
            )}

            {children}

            {showSearchPricesBlock && (
                <SearchPricesBlock
                    segment={segment}
                    context={context}
                    sort={sort}
                    withActionButton={withActionButton}
                />
            )}
        </article>
    );
};

export default React.memo(SearchSegment);
