import React, {RefObject} from 'react';
import B from 'bem-cn-lite';

import {CHAR_EM_DASH} from '../../../../lib/stringUtils';

import ISegment from '../../../../interfaces/segment/ISegment';
import IStateFlags from '../../../../interfaces/state/flags/IStateFlags';

import {formatLocalTime} from '../../../../lib/date/utils';
import getSubtypeByTransportType from '../../../../lib/station/getSubtypeByTransportType';

import Duration from '../Duration/Duration';
import Station from '../Station/Station';

import timeKeyset from '../../../../i18n/time';

interface ITableIntervalTimeAndStations {
    segment: ISegment;
    isStationsVisible: boolean;
    flags: IStateFlags;

    departureChildrenRef?: RefObject<HTMLDivElement>;
    arrivalChildrenRef?: RefObject<HTMLDivElement>;
    stationOpacity?: 0 | 1;
}

const b = B('TableIntervalTimeAndStations');

const TableIntervalTimeAndStations: React.FC<ITableIntervalTimeAndStations> = ({
    flags,
    segment,
    isStationsVisible,

    departureChildrenRef,
    arrivalChildrenRef,
    stationOpacity,
}) => {
    const {
        thread,
        transport: {code: transportType},
        duration,
        stationTo,
        stationFrom,
    } = segment;
    const stationSubtype = getSubtypeByTransportType(transportType);

    return (
        <td className={b()}>
            <div className={b('times')}>
                <div className={b('time')}>
                    {thread &&
                        thread.beginTime &&
                        thread.endTime &&
                        timeKeyset('interval', {
                            start: formatLocalTime(thread.beginTime),
                            end: formatLocalTime(thread.endTime),
                        })}
                </div>

                <Duration
                    className={b('duration')}
                    flags={flags}
                    duration={duration}
                    description={timeKeyset('on-way')}
                />
            </div>

            <div className={b('stations')} style={{opacity: stationOpacity}}>
                <Station
                    className={b('station')}
                    station={stationFrom}
                    stationIsVisible={isStationsVisible}
                    stationSubtype={stationSubtype}
                    stationRef={departureChildrenRef}
                    dataNoSnippet
                />
                {` ${CHAR_EM_DASH} `}
                <Station
                    className={b('station')}
                    station={stationTo}
                    stationIsVisible={isStationsVisible}
                    stationSubtype={stationSubtype}
                    stationRef={arrivalChildrenRef}
                    dataNoSnippet
                />
            </div>

            <div className={b('density')}>{thread?.density}</div>
        </td>
    );
};

export default TableIntervalTimeAndStations;
