import {React, B} from '../../../base';

import {memo, ReactElement} from 'react';
import {Moment} from 'moment';

import ISegmentBadges from '../../../../interfaces/segment/ISegmentBadges';
import EndpointData from '../../../../interfaces/lib/segments/EndpointData';
import SegmentEventType from '../../../../interfaces/segment/SegmentEventType';
import ISegment from '../../../../interfaces/segment/ISegment';
import ISegmentThreadBase from '../../../../interfaces/segment/ISegmentThreadBase';

import CanceledSegmentBadge from '../../../CanceledSegmentBadge/CanceledSegmentBadge.mobile';

import timeKeyset from '../../../../i18n/time';
import badgesKeyset from '../../../../i18n/badges';

const b = B('SegmentLabel');

const BLUE_BADGES = ['fast', 'fastest'];

interface ISegmentLabel {
    isGone: boolean;
    isInterval: boolean;
    badges: ISegmentBadges;
    endpointData: EndpointData;
    currentTimeMoment: Moment;
    departureMoment: Moment;

    density?: string; // из информации о нитке ("маршрутное такси раз в 10-20 минут")
    cancelType?: ISegment['cancelType'];
    cancelledSegments?: ISegmentThreadBase['cancelledSegments'];
    className?: string;
}

function SegmentLabel({
    isGone,
    isInterval,
    departureMoment,
    badges,
    endpointData,
    currentTimeMoment,

    density,
    cancelType,
    cancelledSegments,
    className,
}: ISegmentLabel): ReactElement {
    if (cancelType) {
        return (
            <CanceledSegmentBadge
                cancelType={cancelType}
                cancelledSegments={cancelledSegments}
                className={className}
            />
        );
    }

    // обработка интервальных рейсов
    if (isInterval && density) {
        return <div className={b({color: 'gray'}, className)}>{density}</div>;
    }

    // обработка данных о прибытии и опозданиях
    if (
        endpointData &&
        !endpointData.isDelay &&
        endpointData.eventType === SegmentEventType.fact
    ) {
        // ушел по расписанию
        return (
            <div className={b({color: 'green'}, className)}>
                {endpointData.text}
            </div>
        );
    }

    const minutesBeforeDeparture = departureMoment.diff(
        currentTimeMoment,
        'minutes',
    );
    const displayMinutesBeforeDeparture =
        !isGone && minutesBeforeDeparture >= 0 && minutesBeforeDeparture < 60;

    const endpointText = [
        displayMinutesBeforeDeparture &&
            (minutesBeforeDeparture === 0
                ? timeKeyset('departing-now')
                : timeKeyset('in-time', {
                      time: timeKeyset('minutes-short', {
                          count: minutesBeforeDeparture,
                      }),
                  })),
        // информация об возможных опозданиях
        endpointData &&
            !endpointData.isGone &&
            endpointData.isDelay &&
            endpointData.text,
    ]
        .filter(Boolean)
        .join(', ');

    // обработка бэйджиков
    const badgesKeys = Object.entries(badges)
        .filter(([, value]) => Boolean(value))
        .map(([key]) => key)
        .sort();

    const greenBadges = badgesKeys.filter(
        badge => !BLUE_BADGES.includes(badge),
    );
    const blueBadges = badgesKeys.filter(badge => BLUE_BADGES.includes(badge));

    return (
        <>
            {Boolean(endpointText) && (
                <div className={b({color: 'red'}, className)}>
                    {endpointText}
                </div>
            )}
            {getBadge(greenBadges, 'green', className)}
            {getBadge(blueBadges, 'blue', className)}
        </>
    );
}

function getBadge(
    badgesKeys: string[],
    color: string,
    mixin?: string,
): ReactElement | null {
    if (badgesKeys.length) {
        return (
            <div className={b({color}, mixin)}>
                {badgesKeys
                    .map((key): string => badgesKeyset.get(key))
                    .filter(Boolean)
                    .map(text => (
                        <span className={b('badge')} key={text}>
                            {text}
                        </span>
                    ))}
            </div>
        );
    }

    return null;
}

export default memo(SegmentLabel);
