import React, {PureComponent, RefObject} from 'react';
import B from 'bem-cn-lite';
import {momentTimezone as moment} from '../../../../../reexports';

import {HOURS} from '../../../../lib/date/constants';

import IStateFlags from '../../../../interfaces/state/flags/IStateFlags';
import ISearchContext from '../../../../interfaces/state/search/ISearchContext';

import {isCity} from '../../../../lib/point/pointType';
import {getDuration} from '../../utilities/getDuration';
import {isRangeArrival} from '../../utilities/isRangeArrival';
import getSubtypeByTransportType from '../../../../lib/station/getSubtypeByTransportType';

import Time from '../Time/Time';
import Point from '../Point/Point';
import Duration from '../Duration/Duration';
import RailwayTime from '../../../RailwayTime/RailwayTime';

import timeKeyset from '../../../../i18n/time';
import segmentKeyset from '../../../../i18n/segment';

interface ITableTimeAndStationsProps {
    segment: any;
    context: ISearchContext;
    showArrivalDate: boolean;
    flags: IStateFlags;
    showStops: boolean;
    showDepartureDate: boolean;
    isStationsVisible: boolean;

    departureChildrenRef?: RefObject<HTMLDivElement>;
    arrivalChildrenRef?: RefObject<HTMLDivElement>;
    stationOpacity?: 0 | 1;
}

const defaultDurationOptions = {};
const rangeDurationOptions = {
    shortestUnit: HOURS,
};

const b = B('TableTimeAndStations');

export default class TableTimeAndStations extends PureComponent<ITableTimeAndStationsProps> {
    getArrival(rangeArrival): React.ReactNode {
        const {segment, showArrivalDate} = this.props;
        const {minArrival, maxArrival, stationTo, isFuzzyTo, arrivalMoment} =
            segment;

        if (!rangeArrival) {
            return (
                <Time
                    b={b}
                    timeMoment={arrivalMoment}
                    showDate={showArrivalDate}
                    isFuzzy={isFuzzyTo}
                />
            );
        }

        return (
            <div className={b('arrivalRange')}>
                <Time
                    b={b}
                    timeMoment={moment.tz(minArrival, stationTo.timezone)}
                    showDate={showArrivalDate}
                    isFuzzy={isFuzzyTo}
                />
                ...
                <Time
                    b={b}
                    timeMoment={moment.tz(maxArrival, stationTo.timezone)}
                    showDate={showArrivalDate}
                    isFuzzy={isFuzzyTo}
                />
            </div>
        );
    }

    render(): React.ReactNode {
        const {
            flags,
            context,
            segment,

            showStops,
            showDepartureDate,
            showArrivalDate,
            isStationsVisible,
            departureChildrenRef,
            arrivalChildrenRef,
            stationOpacity,
        } = this.props;

        const {
            departureMoment,
            departureRailwayMoment,
            arrivalMoment,
            arrivalRailwayMoment,
            stationFrom,
            stationTo,
            isMetaSegment,
            isFuzzyFrom,
            transport,
            thread,

            stops,
            useCityInsteadStationTo,
            useCityInsteadStationFrom,
        } = segment;

        const transportType = transport.code;
        const stationSubtype = getSubtypeByTransportType(transportType);
        const isExpress = thread && (thread.isExpress || thread.isAeroExpress);
        const rangeArrival = isRangeArrival(segment);

        const {when, from, to} = context;
        const duration = getDuration(segment);
        const withDate = showDepartureDate || showArrivalDate;

        const fromPointRows =
            Number(Boolean(isStationsVisible)) +
            Number(Boolean(stationFrom.platform));
        const toPointRows =
            Number(Boolean(isStationsVisible)) +
            Number(Boolean(stationTo.platform));

        return (
            <>
                <td
                    className={b('departure', {
                        withDate: withDate && !showDepartureDate,
                    })}
                >
                    <Time
                        b={b}
                        timeMoment={departureMoment}
                        showDate={showDepartureDate}
                        isFuzzy={isFuzzyFrom}
                    />

                    <RailwayTime
                        className={b('railwayTime')}
                        timeMoment={departureMoment}
                        railwayMoment={departureRailwayMoment}
                        railwayTimezone={stationFrom.railwayTimezone}
                        showDate={showDepartureDate}
                        when={when}
                    />

                    <Point
                        b={b}
                        station={stationFrom}
                        city={isCity(from) ? from : null}
                        stationIsVisible={isStationsVisible}
                        useCityInsteadStation={useCityInsteadStationFrom}
                        stationSubtype={stationSubtype}
                        pointRef={departureChildrenRef}
                        stationOpacity={stationOpacity}
                        dataNoSnippet
                    />

                    {stops && showStops && (
                        <div
                            className={b('stops', {
                                fakeRows: toPointRows - fromPointRows,
                            })}
                        >
                            {stops}
                        </div>
                    )}
                </td>

                <td
                    className={b('arrival', {
                        withDate: withDate && !showArrivalDate,
                    })}
                >
                    {this.getArrival(rangeArrival)}

                    <RailwayTime
                        className={b('railwayTime')}
                        timeMoment={arrivalMoment}
                        railwayMoment={arrivalRailwayMoment}
                        railwayTimezone={stationTo.railwayTimezone}
                        showDate={showArrivalDate}
                        when={when}
                    />

                    <Point
                        b={b}
                        station={stationTo}
                        city={isCity(to) ? to : null}
                        stationIsVisible={isStationsVisible}
                        useCityInsteadStation={useCityInsteadStationTo}
                        stationSubtype={stationSubtype}
                        pointRef={arrivalChildrenRef}
                        stationOpacity={stationOpacity}
                        dataNoSnippet
                    />

                    {isMetaSegment && rangeArrival && (
                        <div className={b('metaSegmentInfo')}>
                            <div className={b('arrivalRangeDisclaimer')}>
                                {segmentKeyset('arrival-range')}
                            </div>
                        </div>
                    )}
                </td>

                <td className={b('duration', {withDate})}>
                    <Duration
                        flags={flags}
                        duration={duration}
                        transport={transport}
                        isExpress={isExpress}
                        durationOpts={
                            rangeArrival
                                ? rangeDurationOptions
                                : defaultDurationOptions
                        }
                        prefix={
                            rangeArrival
                                ? timeKeyset('duration-from')
                                : undefined
                        }
                    />
                </td>
            </>
        );
    }
}
