import {PureComponent, ReactNode} from 'react';

import {SORTING_TYPES} from '../../lib/sort/utils';

import isSegment from '../../lib/segments/isSegment';
import {getMomentDate} from '../../lib/date/utils';
import getPreviewVector from '../../lib/segments/getPreviewVector';
import getDatesVisibility from '../../lib/segments/getDatesVisibility';
import getStationsVisibility from '../../lib/segments/getStationsVisibility';
import {FilterTransportType, TransportType} from '../../lib/transportType';
import buildSegmentRenderingData from '../../lib/segments/buildSegmentRenderingData';
import getNumberOfDirectSegments from '../../lib/segments/getNumberOfDirectSegments';
import isRidesharingAvailableForDisplay from '../../lib/segments/ridesharing/isRidesharingAvailableForDisplay';
import ISearchContext from 'common/interfaces/state/search/ISearchContext';
import {TBannerInfo} from 'common/interfaces/state/search/BannerInfo';
import IStateFlags from 'common/interfaces/state/flags/IStateFlags';
import IBlablacar from 'common/interfaces/state/blablacar/IBlablacar';
import SearchSegment from 'common/interfaces/state/search/SearchSegment';
import ISearchQuerying from 'common/interfaces/state/search/ISearchQuerying';
import ISearchFiltering from 'common/interfaces/state/search/ISearchFiltering';
import Dispatch from 'common/interfaces/actions/Dispatch';
import Lang from 'common/interfaces/Lang';
import ISearchSort from 'common/interfaces/state/search/ISearchSort';
import IStateEnvironment from 'common/interfaces/state/IStateEnvironment';
import ISearchArchivalData from 'common/interfaces/state/search/ISearchArchivalData';
import IStateUserBrowser from 'common/interfaces/state/user/IStateUserBrowser';
import FilteredSegmentIndexes from 'common/interfaces/lib/segments/FilteredSegmentIndixes';

const isMobile = process.env.PLATFORM === 'mobile';

const DIRECT_STEP = 10;
const BANNER_POSITION = 2;
const SNIPPETS_FOR_DISPLAY_DIRECT = 6;

interface ISearchSegmentsBaseProps {
    flags: IStateFlags;
    vector: number[];
    context: ISearchContext;
    blablacar: IBlablacar;
    segments: SearchSegment[];
    querying: ISearchQuerying;
    filtering: ISearchFiltering;
    dispatch: Dispatch;
    language: Lang;
    sort: ISearchSort;
    filteredSegmentIndices: FilteredSegmentIndexes;
    environment: IStateEnvironment;
    transportTypes: TransportType[];
    isSuburbanSearchResult: boolean;
    archivalData?: ISearchArchivalData;
    browser: IStateUserBrowser;
    queryingPrices?: boolean;
    hideAds?: boolean;
    bannerInfo?: TBannerInfo;
}

export default abstract class SearchSegmentsBase extends PureComponent<ISearchSegmentsBaseProps> {
    abstract getBanner(insertIndex: number): ReactNode;
    abstract getSearchSegment(data: any): ReactNode;
    abstract getDirectSnippet(index: number): ReactNode;
    abstract getGoneFilter(): ReactNode;
    abstract getDatesSeparator(data: any, insertIndex: number): ReactNode;

    canShowRidesharingSegment(): ReactNode {
        const {context, blablacar, flags} = this.props;

        return isRidesharingAvailableForDisplay(context, blablacar, flags);
    }

    /*
     * Собираем массив сниппетов
     * На отрисовку отдаются все сниппеты, вне зависимости от их видимости.
     * Это сделано с целью уменьшения времени отклика на действия пользователя.
     */
    getRenderMap(): any[] {
        const {
            sort,
            flags,
            context,
            dispatch,
            segments,
            filtering,
            environment,
            isSuburbanSearchResult,
            filteredSegmentIndices,
        } = this.props;

        const currentTimeMoment = getMomentDate(
            context.time.now,
            context.from.timezone,
        );
        const vector = getPreviewVector({
            environment,
            vector: this.props.vector,
            filteredSegmentIndices,
        });
        const datesVisibility = getDatesVisibility({
            vector,
            filteredSegmentIndices,
            segments,
            context,
            sort,
        });
        const stationsVisibility = getStationsVisibility({
            vector,
            filteredSegmentIndices,
            segments,
            context,
        });
        const numberOfDirectSegments = getNumberOfDirectSegments(
            vector,
            filteredSegmentIndices,
            segments,
        );

        return this.insertSnippets(
            vector.map(index => {
                const segment = segments[index];

                return {
                    componentBuilder: this.getSearchSegment,
                    data: {
                        sort,
                        flags,
                        context,
                        dispatch,
                        environment,
                        currentTimeMoment,
                        segment: buildSegmentRenderingData(segment, filtering),
                        filtering,
                        isSuburbanSearchResult,
                        ...datesVisibility[index],
                        key: segment.segmentId,
                        isHidden:
                            !(isSegment(segment)
                                ? segment.isArchival
                                : false) && !filteredSegmentIndices[index],
                        isStationsVisible: stationsVisibility[index],
                        showStops:
                            context.transportType !== FilterTransportType.all ||
                            isSuburbanSearchResult,
                        numberOfDirectSegments,
                    },
                };
            }),
        );
    }

    /**
     * Метод для вставки в выдачу различных сниппетов: блаблакар, директ,
     * фильтр по ушедшим рейсам, разделитель суток
     * Поиск места вставки происходит среди видимых сниппетов. После того, как
     * найден элемент, перед которым разместится блок, мы производим вставку в
     * соответствующее место в массиве всех сниппетов.
     * @param snippets - массив сегментов
     */
    insertSnippets(snippets: any[]): any[] {
        const visibleSnippets = snippets.filter(({data}) => !data.isHidden);

        if (visibleSnippets.length) {
            snippets = this.insertBanner(snippets, visibleSnippets);
            snippets = this.insertDirect(snippets, visibleSnippets);
            snippets = this.insertGoneFilter(snippets, visibleSnippets);
            snippets = this.insertDatesSeparator(snippets, visibleSnippets);
        }

        return snippets.filter(Boolean);
    }

    /*
     * Метод вставки баннера.
     * @param {Object[]} snippets - массив всех сниппетов
     * @param {Object[]} visibleSnippets - массив видимых сниппетов
     */
    insertBanner(snippets: any[], visibleSnippets: any[]): any[] {
        const target = visibleSnippets[BANNER_POSITION];
        const insertIndex = target ? snippets.indexOf(target) : snippets.length;

        snippets.splice(insertIndex, 0, this.getBanner(insertIndex));

        return snippets;
    }

    /*
     * Метод вставки блоков директа
     * @param {Object[]} snippets - массив всех сниппетов
     * @param {Object[]} visibleSnippets - массив видимых сниппетов
     */
    insertDirect(snippets: any[], visibleSnippets: any[]): any[] {
        for (let i = 0; i < visibleSnippets.length; i += DIRECT_STEP) {
            const chunk = visibleSnippets.slice(i, i + DIRECT_STEP);

            if (chunk.length > SNIPPETS_FOR_DISPLAY_DIRECT) {
                const target = visibleSnippets[i + DIRECT_STEP];

                // Для тача не вставляем рекламный сниппет в конец выдачи,
                // потому в нём что сразу после неё идёт реклама в футере
                if (target || !isMobile) {
                    const insertIndex = target
                        ? snippets.indexOf(target)
                        : snippets.length;

                    snippets.splice(insertIndex, 0, this.getDirectSnippet(i));
                }
            }
        }

        return snippets;
    }

    /*
     * Метод вставки фильтра по ушедшим рейсам
     * @param {Object[]} snippets - массив всех сниппетов
     * @param {Object[]} visibleSnippets - массив видимых сниппетов
     */
    insertGoneFilter(snippets: any[], visibleSnippets: any[]): any[] {
        const {filtering, sort} = this.props;
        const {gone} = filtering.filters || {};
        const {by, reverse} = sort;

        if (gone?.availableWithActiveOptions) {
            const target = visibleSnippets.find(
                ({data}) =>
                    !data.segment.isInterval &&
                    (by !== SORTING_TYPES.DEPARTURE ||
                        data.segment.isGone === reverse),
            );
            const insertIndex = target
                ? snippets.indexOf(target)
                : snippets.length;

            snippets.splice(insertIndex, 0, this.getGoneFilter());
        }

        return snippets;
    }

    /*
     * Метод вставки разделителя суток
     * @param {Object[]} snippets - массив всех сниппетов
     * @param {Object[]} visibleSnippets - массив видимых сниппетов
     */
    insertDatesSeparator(snippets: any[], visibleSnippets: any[]): any[] {
        visibleSnippets.forEach(({data}, index) => {
            if (data.showDatesSeparator) {
                const insertIndex = snippets.indexOf(visibleSnippets[index]);

                snippets.splice(
                    insertIndex,
                    0,
                    this.getDatesSeparator(data, insertIndex),
                );
            }
        });

        return snippets;
    }
}
