import React, {ReactNode} from 'react';
import B from 'bem-cn-lite';

import {EBannerType} from '../../../common/interfaces/state/search/BannerInfo';

import isLastochkaPage from '../../lib/search/isLastochkaPage';
import {getSearchTitle} from '../../lib/search/getSearchTitle';
import {anyFiltersAvailable} from '../../lib/filters/filtering';

import LocalTime from '../LocalTime/LocalTime';
import RidesharingSegment from '../RidesharingSegment';
import SearchSegmentsBase from './SearchSegments.base';
import FilterGone from '../FilterGone/FilterGone.mobile';
import FiltersToolbar from '../FiltersToolbar/FiltersToolbar';
import SearchSegment from '../SearchSegment/SearchSegment.mobile';
import SearchTransfer from '../SearchTransfer/SearchTransfer.mobile';
import DatesSeparator from '../DatesSeparator/DatesSeparator.mobile';
import SearchMetaSegment from '../SearchMetaSegment/SearchMetaSegment.mobile';
import TravelHotelsBanner from '../TravelHotelsBanner/TravelHotelsBanner.mobile';
import SearchSegmentsDirect from './components/SearchSegmentsDirect/SearchSegmentsDirect';

import keyset from '../../i18n/search';
import notFoundKeyset from '../../i18n/search-title-notfound';

const b = B('SearchSegments');

export default class MobileSearchSegments extends SearchSegmentsBase {
    getSearchSegment(data): ReactNode {
        const {segment} = data;
        let SegmentComponent:
            | typeof SearchSegment
            | typeof SearchTransfer
            | typeof SearchMetaSegment = SearchSegment;

        if (segment.isTransfer) {
            SegmentComponent = SearchTransfer;
        } else if (segment.isMetaSegment) {
            SegmentComponent = SearchMetaSegment;
        }

        return (
            <SegmentComponent
                key={segment.segmentId}
                data={data}
                className={b('segment')}
            />
        );
    }

    getDirectSnippet(index): ReactNode {
        const {context, transportTypes} = this.props;

        return (
            <SearchSegmentsDirect
                index={index}
                key={`direct-${index}`}
                context={context}
                transportTypes={transportTypes}
            />
        );
    }

    getGoneFilter(): ReactNode {
        const {filtering, dispatch, language, archivalData} = this.props;

        if (archivalData || !filtering.filters?.gone) {
            return null;
        }

        return (
            <FilterGone
                {...filtering.filters?.gone}
                key="goneFilter"
                dispatch={dispatch}
                language={language}
                className={b('filterGone')}
            />
        );
    }

    getDatesSeparator({segment}, insertIndex): ReactNode {
        const {context, sort} = this.props;

        return (
            <DatesSeparator
                key={`date-${insertIndex}`}
                date={segment.departure}
                endOfDay={sort.reverse}
                timezone={context.from.timezone}
            />
        );
    }

    getBanner(insertIndex): ReactNode {
        const {blablacar, language, flags, bannerInfo} = this.props;

        switch (bannerInfo?.bannerType) {
            case EBannerType.YA_TRAVEL_HOTELS:
                return (
                    <TravelHotelsBanner
                        key="hotelsBanner"
                        className={b('banner')}
                        bannerInfo={bannerInfo}
                    />
                );
            case EBannerType.BLA_BLA_CAR:
                return (
                    <RidesharingSegment
                        {...blablacar}
                        key={`bbc-${insertIndex}`}
                        language={language}
                        className={b('blablacar')}
                        flags={flags}
                    />
                );
            case EBannerType.ADVERTISING: {
                return this.getDirectSnippet(-1);
            }

            default:
                return null;
        }
    }

    renderFilters(): ReactNode {
        const {
            filtering,
            sort,
            segments,
            transportTypes,
            queryingPrices,
            language,
        } = this.props;

        if (!anyFiltersAvailable(filtering)) {
            return null;
        }

        return (
            <div className={b('filter')}>
                <LocalTime className={b('localTime')} />
                <FiltersToolbar
                    sort={sort}
                    segments={segments}
                    filtering={filtering}
                    transportTypes={transportTypes}
                    queryingPrices={queryingPrices}
                    language={language}
                />
            </div>
        );
    }

    renderTitle(): ReactNode {
        const {context, filtering, archivalData} = this.props;
        const {transportType, when, from, to} = context;

        let title;
        const titleParams = {
            from: from.title,
            to: to.title,
        };

        if (archivalData) {
            title = notFoundKeyset('archival-flights-title', titleParams);
        } else if (isLastochkaPage(transportType, when, filtering)) {
            title = keyset('title-lastochka', titleParams);
        } else {
            title = getSearchTitle(context);
        }

        return archivalData ? (
            <div className={b('archivalTitle')}>{title}</div>
        ) : (
            <h1 className={b('title')}>{title}</h1>
        );
    }

    render(): ReactNode {
        return (
            <section className={b()}>
                {this.renderTitle()}
                {this.renderFilters()}
                {this.getRenderMap().map(snippet =>
                    snippet.componentBuilder
                        ? snippet.componentBuilder(snippet.data)
                        : snippet,
                )}
            </section>
        );
    }
}
